package sqlite4a;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Arrays;

/**
 * @author Daniel Serdyukov
 */
class SQLiteJdbcRowSet implements SQLiteRowSet {

    private static final byte[] EMPTY_BLOB = new byte[0];

    private static final int[] JDBC_TYPES = new int[]{Types.INTEGER, Types.REAL, Types.BLOB};

    private static final int[] SQLITE_TYPES = new int[]{SQLITE_INTEGER, SQLITE_FLOAT, SQLITE_BLOB};

    private final ResultSet mResultSet;

    SQLiteJdbcRowSet(@NonNull ResultSet resultSet) {
        mResultSet = resultSet;
    }

    @Override
    public boolean step() {
        try {
            return mResultSet.next();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public int getColumnCount() {
        try {
            return mResultSet.getMetaData().getColumnCount();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @NonNull
    @Override
    public String getColumnName(@IntRange(from = 0) int columnIndex) {
        try {
            return mResultSet.getMetaData().getColumnName(columnIndex + 1);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public int getColumnType(@IntRange(from = 0) int columnIndex) {
        try {
            final int columnType = mResultSet.getMetaData().getColumnType(columnIndex + 1);
            final int index = Arrays.binarySearch(JDBC_TYPES, columnType);
            if (index >= 0) {
                return SQLITE_TYPES[index];
            }
            return SQLITE_TEXT;
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public long getColumnLong(@IntRange(from = 0) int columnIndex) {
        try {
            return mResultSet.getLong(columnIndex + 1);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public double getColumnDouble(@IntRange(from = 0) int columnIndex) {
        try {
            return mResultSet.getDouble(columnIndex + 1);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public String getColumnString(@IntRange(from = 0) int columnIndex) {
        try {
            return mResultSet.getString(columnIndex + 1);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public byte[] getColumnBlob(@IntRange(from = 0) int columnIndex) {
        try {
            final byte[] bytes = mResultSet.getBytes(columnIndex + 1);
            if (bytes == null) {
                return EMPTY_BLOB;
            }
            return bytes;
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

}
