package sqlite4a;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;

/**
 * @author Daniel Serdyukov
 */
@SuppressWarnings("TryFinallyCanBeTryWithResources")
class SQLiteJdbcDb extends SQLiteDb {

    private final Connection mConnection;

    private SQLiteJdbcStmt mLastStmt;

    @VisibleForTesting
    SQLiteJdbcDb(@NonNull Connection connection) {
        mConnection = connection;
    }

    static boolean inClasspath() {
        try {
            Class.forName("org.sqlite.JDBC");
            return true;
        } catch (ClassNotFoundException e) {
            return false;
        }
    }

    @NonNull
    static SQLiteJdbcDb openConnection(@NonNull String path) {
        try {
            final Connection connection = DriverManager.getConnection("jdbc:sqlite:" + path);
            connection.setAutoCommit(true);
            return new SQLiteJdbcDb(connection);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public boolean isReadOnly() {
        System.out.println("SQLiteJdbcDb.isReadOnly " + mConnection);
        try {
            return mConnection.isReadOnly();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public boolean isAutoCommit() {
        try {
            return mConnection.getAutoCommit();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public void trace(@NonNull Logger func) {

    }

    @Override
    public void profile(@NonNull Logger func) {

    }

    @Override
    public void exec(@NonNull String sql, @Nullable Exec func) {
        mLastStmt = new SQLiteJdbcStmt(mConnection, sql);
        try {
            mLastStmt.execute(func);
        } finally {
            mLastStmt.close();
        }
    }

    @NonNull
    @Override
    public SQLiteStmt prepare(@NonNull String sql) {
        return (mLastStmt = new SQLiteJdbcStmt(mConnection, sql));
    }

    @Override
    public int getChanges() {
        if (mLastStmt != null) {
            return mLastStmt.getChanges();
        }
        return 0;
    }

    @Override
    public long getLastInsertRowId() {
        if (mLastStmt != null) {
            return mLastStmt.getLastInsertRowId();
        }
        return -1;
    }

    @Override
    public void close() {
        try {
            mConnection.close();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

}
