package sqlite4a;

import android.support.annotation.IntDef;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.io.Closeable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * @author Daniel Serdyukov
 */
public class SQLiteDb implements Closeable {

    public static final int OPEN_READONLY = 0b1;

    public static final int OPEN_READWRITE = 0b10;

    public static final int OPEN_CREATE = 0b100;

    public static final int OPEN_URI = 0b1000000;

    private final long mDbPtr;

    private SQLiteDb(long dbPtr) {
        mDbPtr = dbPtr;
    }

    public static void loadLibrary() {
        Runtime.getRuntime().loadLibrary("sqlite4a");
    }

    @NonNull
    public static SQLiteDb open(@NonNull String path, @OpenFlags int openFlags) {
        return new SQLiteDb(nativeOpen(path, openFlags));
    }

    private static native long nativeOpen(String path, int openFlags);

    private static native void nativeTrace(long dbPtr, Logger func);

    private static native void nativeProfile(long dbPtr, Logger func);

    private static native boolean nativeIsReadOnly(long dbPtr);

    private static native boolean nativeIsAutoCommit(long dbPtr);

    private static native void nativeExec(long dbPtr, String sql, Exec func);

    private static native long nativePrepare(long dbPtr, String sql);

    @IntRange(from = 0)
    private static native int nativeGetChanges(long dbPtr);

    @IntRange(from = -1)
    private static native long nativeGetLastInsertRowId(long dbPtr);

    private static native void nativeClose(long dbPtr);

    public boolean isReadOnly() {
        return nativeIsReadOnly(mDbPtr);
    }

    public boolean isAutoCommit() {
        return nativeIsAutoCommit(mDbPtr);
    }

    public void trace(@NonNull Logger func) {
        nativeTrace(mDbPtr, func);
    }

    public void profile(@NonNull Logger func) {
        nativeProfile(mDbPtr, func);
    }

    public void exec(@NonNull String sql, @Nullable Exec func) {
        nativeExec(mDbPtr, sql, func);
    }

    @NonNull
    public SQLiteStmt prepare(@NonNull String sql) {
        return new SQLiteStmt(nativePrepare(mDbPtr, sql));
    }

    @IntRange(from = 0)
    public int getChanges() {
        return nativeGetChanges(mDbPtr);
    }

    @IntRange(from = -1)
    public long getLastInsertRowId() {
        return nativeGetLastInsertRowId(mDbPtr);
    }

    @Override
    public void close() {
        nativeClose(mDbPtr);
    }

    public interface Exec {
        void call(String[] values, String[] columns);
    }

    public interface Logger {
        void log(@NonNull String sql, long execTimeMs);
    }

    @IntDef(flag = true, value = {OPEN_READONLY, OPEN_READWRITE, OPEN_CREATE, OPEN_URI})
    @Retention(RetentionPolicy.SOURCE)
    public @interface OpenFlags {
    }

}
