/*
 * Decompiled with CFR 0.152.
 */
package software.xdev.caching;

import java.lang.ref.SoftReference;
import java.time.Clock;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

public class ExpiringLimitedCache<K, V> {
    protected final Duration expirationTime;
    protected final AtomicInteger cleanUpExecutorCounter = new AtomicInteger(1);
    protected final ThreadFactory cleanUpExecutorThreadFactory;
    protected ScheduledExecutorService cleanUpExecutor;
    protected final Object cleanUpExecutorLock = new Object();
    protected Consumer<String> logConsumer;
    protected final Map<K, SoftReference<CacheValue<V>>> cache;

    public ExpiringLimitedCache(String cacheName, Duration expirationTime, int maxSize) {
        this.expirationTime = Objects.requireNonNull(expirationTime);
        if (expirationTime.toSeconds() < 1L) {
            throw new IllegalStateException();
        }
        this.cache = Collections.synchronizedMap(new LimitedLinkedHashMap(maxSize));
        this.cleanUpExecutorThreadFactory = r -> {
            Thread thread = new Thread(r);
            thread.setName(cacheName + "-Cache-Cleanup-Executor-" + this.cleanUpExecutorCounter.getAndIncrement());
            thread.setDaemon(true);
            return thread;
        };
    }

    public void setLogConsumer(Consumer<String> logConsumer) {
        this.logConsumer = logConsumer;
    }

    protected void log(String s) {
        if (this.logConsumer != null) {
            this.logConsumer.accept(s);
        }
    }

    public void put(K key, V value) {
        this.log("put called for key[hashcode=" + key.hashCode() + "]: " + key);
        this.cache.put(key, new SoftReference<CacheValue<V>>(new CacheValue<V>(value, ExpiringLimitedCache.currentUtcTime().plus(this.expirationTime))));
        this.startCleanupExecutorIfRequired();
    }

    public V get(K key) {
        String keyLogStr = "key[hashcode=" + key.hashCode() + "]";
        this.log("get called for " + keyLogStr + ": " + key);
        SoftReference<CacheValue<V>> ref = this.cache.get(key);
        if (ref == null) {
            this.log(keyLogStr + " not in cache");
            return null;
        }
        CacheValue<V> value = ref.get();
        if (value == null) {
            this.log("Value for " + keyLogStr + " was disposed by GC");
            return null;
        }
        if (value.isExpired()) {
            this.log(keyLogStr + " is expired");
            this.cache.remove(key);
            this.shutdownCleanupExecutorIfRequired();
            return null;
        }
        this.log(keyLogStr + " is present");
        return value.value();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void startCleanupExecutorIfRequired() {
        if (this.cleanUpExecutor != null) {
            return;
        }
        Object object = this.cleanUpExecutorLock;
        synchronized (object) {
            if (this.cleanUpExecutor != null) {
                return;
            }
            this.log("Starting cleanupExecutor");
            this.cleanUpExecutor = Executors.newScheduledThreadPool(1, this.cleanUpExecutorThreadFactory);
            this.cleanUpExecutor.scheduleAtFixedRate(this::runCleanup, this.expirationTime.toMillis(), this.expirationTime.toMillis() / 2L, TimeUnit.MILLISECONDS);
        }
    }

    private void runCleanup() {
        long startTime = System.currentTimeMillis();
        List<Object> toClear = this.cache.entrySet().stream().filter(e -> Optional.ofNullable((CacheValue)((SoftReference)e.getValue()).get()).map(CacheValue::isExpired).orElse(true)).map(Map.Entry::getKey).toList();
        toClear.forEach(this.cache::remove);
        this.log("Cleared " + toClear.size() + "x cached entries, took " + (System.currentTimeMillis() - startTime) + "ms");
        this.shutdownCleanupExecutorIfRequired();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void shutdownCleanupExecutorIfRequired() {
        if (this.cache.isEmpty()) {
            Object object = this.cleanUpExecutorLock;
            synchronized (object) {
                this.log("Shutting down cleanupExecutor");
                this.cleanUpExecutor.shutdownNow();
                this.cleanUpExecutor = null;
            }
        }
    }

    public int cacheSize() {
        return this.cache.size();
    }

    protected static LocalDateTime currentUtcTime() {
        return LocalDateTime.now(Clock.systemUTC());
    }

    public static class LimitedLinkedHashMap<K, V>
    extends LinkedHashMap<K, V> {
        protected final int maxSize;

        public LimitedLinkedHashMap(int maxSize) {
            if (maxSize < 1) {
                throw new IllegalStateException();
            }
            this.maxSize = maxSize;
        }

        @Override
        protected boolean removeEldestEntry(Map.Entry<K, V> eldest) {
            return this.size() > this.maxSize;
        }
    }

    public record CacheValue<V>(V value, LocalDateTime utcCacheExpirationTime) {
        public CacheValue {
            Objects.requireNonNull(utcCacheExpirationTime);
        }

        public boolean isExpired() {
            return ExpiringLimitedCache.currentUtcTime().isAfter(this.utcCacheExpirationTime);
        }
    }
}

