/*
 * Copyright © 2023 XDEV Software (https://xdev.software)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package software.xdev.chartjs.model.options.scales;

import java.math.BigDecimal;

import software.xdev.chartjs.model.enums.ScalesPosition;
import software.xdev.chartjs.model.javascript.JavaScriptFunction;
import software.xdev.chartjs.model.options.ticks.Ticks;


public class Scale<T extends Ticks<T>, S extends Scale<T, S>>
{
	protected T ticks;
	protected Boolean display;
	protected ScalesPosition position;
	protected BigDecimal min;
	protected BigDecimal max;
	protected BigDecimal suggestedMin;
	protected BigDecimal suggestedMax;
	protected String stack;
	protected BigDecimal stackWeight;
	protected JavaScriptFunction beforeUpdate;
	protected JavaScriptFunction beforeSetDimensions;
	protected JavaScriptFunction afterSetDimensions;
	protected JavaScriptFunction beforeDataLimits;
	protected JavaScriptFunction afterDataLimits;
	protected JavaScriptFunction beforeBuildTicks;
	protected JavaScriptFunction afterBuildTicks;
	protected JavaScriptFunction beforeTickToLabelConversion;
	protected JavaScriptFunction afterTickToLabelConversion;
	protected JavaScriptFunction beforeCalculateTickRotation;
	protected JavaScriptFunction afterCalculateTickRotation;
	protected JavaScriptFunction beforeFit;
	protected JavaScriptFunction afterFit;
	protected JavaScriptFunction afterUpdate;
	protected GridLines gridLines;
	protected ScaleTitle title;
	protected Boolean stacked;
	
	/**
	 * @see #setTicks(Ticks)
	 */
	public T getTicks()
	{
		return this.ticks;
	}
	
	/**
	 * Defines options for the tick marks that are generated by the axis
	 */
	public S setTicks(final T ticks)
	{
		this.ticks = ticks;
		return this.self();
	}
	
	public BigDecimal getMin()
	{
		return this.min;
	}
	
	public S setMin(final BigDecimal min)
	{
		this.min = min;
		return this.self();
	}
	
	public BigDecimal getMax()
	{
		return this.max;
	}
	
	public S setMax(final BigDecimal max)
	{
		this.max = max;
		return this.self();
	}
	
	public BigDecimal getSuggestedMin()
	{
		return this.suggestedMin;
	}
	
	public S setSuggestedMin(final BigDecimal suggestedMin)
	{
		this.suggestedMin = suggestedMin;
		return this.self();
	}
	
	public BigDecimal getSuggestedMax()
	{
		return this.suggestedMax;
	}
	
	public S setSuggestedMax(final BigDecimal suggestedMax)
	{
		this.suggestedMax = suggestedMax;
		return this.self();
	}
	
	/**
	 * @see #setDisplay(Boolean)
	 */
	public Boolean getDisplay()
	{
		return this.display;
	}
	
	/**
	 * <p>
	 * If true, show the scale including gridlines, ticks, and labels. Overrides gridLines.display, scaleLabel.display,
	 * and ticks.display.
	 * </p>
	 *
	 * <p>
	 * Default {@code true}
	 * </p>
	 */
	public S setDisplay(final Boolean display)
	{
		this.display = display;
		return this.self();
	}
	
	/**
	 * @see #setPosition(ScalesPosition)
	 */
	public ScalesPosition getPosition()
	{
		return this.position;
	}
	
	/**
	 * <p>
	 * Position of the scale. Possible values are 'top', 'left', 'bottom' and 'right'.
	 * </p>
	 *
	 * <p>
	 * Default {@code "left"}
	 * </p>
	 */
	public S setPosition(final ScalesPosition position)
	{
		this.position = position;
		return this.self();
	}
	
	/**
	 * @see #setBeforeUpdate(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeUpdate()
	{
		return this.beforeUpdate;
	}
	
	/**
	 * <p>
	 * Callback called before the update process starts. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeUpdate(final JavaScriptFunction beforeUpdate)
	{
		this.beforeUpdate = beforeUpdate;
		return this.self();
	}
	
	/**
	 * @see #setBeforeSetDimensions(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeSetDimensions()
	{
		return this.beforeSetDimensions;
	}
	
	/**
	 * <p>
	 * Callback that runs before dimensions are set. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeSetDimensions(final JavaScriptFunction beforeSetDimensions)
	{
		this.beforeSetDimensions = beforeSetDimensions;
		return this.self();
	}
	
	/**
	 * @see #setAfterSetDimensions(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterSetDimensions()
	{
		return this.afterSetDimensions;
	}
	
	/**
	 * <p>
	 * Callback that runs after dimensions are set. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterSetDimensions(final JavaScriptFunction afterSetDimensions)
	{
		this.afterSetDimensions = afterSetDimensions;
		return this.self();
	}
	
	/**
	 * @see #setBeforeDataLimits(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeDataLimits()
	{
		return this.beforeDataLimits;
	}
	
	/**
	 * <p>
	 * Callback that runs before data limits are determined. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeDataLimits(final JavaScriptFunction beforeDataLimits)
	{
		this.beforeDataLimits = beforeDataLimits;
		return this.self();
	}
	
	/**
	 * @see #setAfterDataLimits(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterDataLimits()
	{
		return this.afterDataLimits;
	}
	
	/**
	 * <p>
	 * Callback that runs after data limits are determined. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterDataLimits(final JavaScriptFunction afterDataLimits)
	{
		this.afterDataLimits = afterDataLimits;
		return this.self();
	}
	
	/**
	 * @see #setBeforeBuildTicks(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeBuildTicks()
	{
		return this.beforeBuildTicks;
	}
	
	/**
	 * <p>
	 * Callback that runs before ticks are created. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeBuildTicks(final JavaScriptFunction beforeBuildTicks)
	{
		this.beforeBuildTicks = beforeBuildTicks;
		return this.self();
	}
	
	/**
	 * @see #setAfterBuildTicks(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterBuildTicks()
	{
		return this.afterBuildTicks;
	}
	
	/**
	 * <p>
	 * Callback that runs after ticks are created. Useful for filtering ticks. Passed a single argument, the scale
	 * instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterBuildTicks(final JavaScriptFunction afterBuildTicks)
	{
		this.afterBuildTicks = afterBuildTicks;
		return this.self();
	}
	
	/**
	 * @see #setBeforeTickToLabelConversion(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeTickToLabelConversion()
	{
		return this.beforeTickToLabelConversion;
	}
	
	/**
	 * <p>
	 * Callback that runs before ticks are converted into strings. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeTickToLabelConversion(final JavaScriptFunction beforeTickToLabelConversion)
	{
		this.beforeTickToLabelConversion = beforeTickToLabelConversion;
		return this.self();
	}
	
	/**
	 * @see #setAfterTickToLabelConversion(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterTickToLabelConversion()
	{
		return this.afterTickToLabelConversion;
	}
	
	/**
	 * <p>
	 * Callback that runs after ticks are converted into strings. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterTickToLabelConversion(final JavaScriptFunction afterTickToLabelConversion)
	{
		this.afterTickToLabelConversion = afterTickToLabelConversion;
		return this.self();
	}
	
	/**
	 * @see #setBeforeCalculateTickRotation(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeCalculateTickRotation()
	{
		return this.beforeCalculateTickRotation;
	}
	
	/**
	 * <p>
	 * Callback that runs before tick rotation is determined. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeCalculateTickRotation(final JavaScriptFunction beforeCalculateTickRotation)
	{
		this.beforeCalculateTickRotation = beforeCalculateTickRotation;
		return this.self();
	}
	
	/**
	 * @see #setAfterCalculateTickRotation(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterCalculateTickRotation()
	{
		return this.afterCalculateTickRotation;
	}
	
	/**
	 * <p>
	 * Callback that runs after tick rotation is determined. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterCalculateTickRotation(final JavaScriptFunction afterCalculateTickRotation)
	{
		this.afterCalculateTickRotation = afterCalculateTickRotation;
		return this.self();
	}
	
	/**
	 * @see #setBeforeFit(JavaScriptFunction)
	 */
	public JavaScriptFunction getBeforeFit()
	{
		return this.beforeFit;
	}
	
	/**
	 * <p>
	 * Callback that runs before the scale fits to the canvas. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setBeforeFit(final JavaScriptFunction beforeFit)
	{
		this.beforeFit = beforeFit;
		return this.self();
	}
	
	/**
	 * @see #setAfterFit(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterFit()
	{
		return this.afterFit;
	}
	
	/**
	 * <p>
	 * Callback that runs after the scale fits to the canvas. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterFit(final JavaScriptFunction afterFit)
	{
		this.afterFit = afterFit;
		return this.self();
	}
	
	/**
	 * @see #setAfterUpdate(JavaScriptFunction)
	 */
	public JavaScriptFunction getAfterUpdate()
	{
		return this.afterUpdate;
	}
	
	/**
	 * <p>
	 * Callback that runs at the end of the update process. Passed a single argument, the scale instance.
	 * </p>
	 *
	 * <p>
	 * Default {@code undefined}
	 * </p>
	 */
	public S setAfterUpdate(final JavaScriptFunction afterUpdate)
	{
		this.afterUpdate = afterUpdate;
		return this.self();
	}
	
	/**
	 * @see #setGridLines(GridLines)
	 */
	public GridLines getGridLines()
	{
		return this.gridLines;
	}
	
	/**
	 * <p>
	 * See grid line configuration section.
	 * </p>
	 *
	 * <p>
	 * Default {@code -}
	 * </p>
	 */
	public S setGridLines(final GridLines gridLines)
	{
		this.gridLines = gridLines;
		return this.self();
	}
	
	/**
	 * @see #setTitle(ScaleTitle)
	 */
	public ScaleTitle getTitle()
	{
		return this.title;
	}
	
	/**
	 * <p>
	 * See scale title configuration section.
	 * </p>
	 *
	 * <p>
	 * Default {@code }
	 * </p>
	 */
	public S setTitle(final ScaleTitle title)
	{
		this.title = title;
		return this.self();
	}
	
	public String getStack()
	{
		return this.stack;
	}
	
	public S setStack(final String stack)
	{
		this.stack = stack;
		return this.self();
	}
	
	public BigDecimal getStackWeight()
	{
		return this.stackWeight;
	}
	
	public S setStackWeight(final BigDecimal stackWeight)
	{
		this.stackWeight = stackWeight;
		return this.self();
	}
	
	/**
	 * @see #setStacked(Boolean stacked)
	 */
	public Boolean getStacked()
	{
		return this.stacked;
	}
	
	/**
	 * If true, bars are stacked on the x-axis
	 */
	public S setStacked(final Boolean stacked)
	{
		this.stacked = stacked;
		return this.self();
	}
	
	protected S self()
	{
		return (S)this;
	}
}
