package software.kes.kraftwerk;

import com.jnape.palatable.lambda.adt.Maybe;
import com.jnape.palatable.lambda.adt.coproduct.CoProduct2;
import com.jnape.palatable.lambda.functions.Fn1;
import software.kes.collectionviews.ImmutableVector;
import software.kes.collectionviews.Vector;
import software.kes.enhancediterables.ImmutableIterable;

import static com.jnape.palatable.lambda.adt.Maybe.just;
import static com.jnape.palatable.lambda.adt.Maybe.nothing;
import static software.kes.kraftwerk.Generators.constant;

/**
 * A builder that facilitates creating {@link Generator}s for complex {@link String}s.
 * <p>
 * After instantiating a {@code StringGeneratorBuilder}, you can call the various {@code add} methods to
 * specify "segments" of the final output, and then call {@link StringGeneratorBuilder#build} to create the
 * {@code Generator}.  When the {@code Generator} is invoked, all segments will be joined together in the order they were
 * specified.
 * <p>
 * A {@code StringGeneratorBuilder} is immutable and can be used to construct multiple {@code Generator}s.
 * All methods that alter a {@code StringGeneratorBuilder} return a new {@code StringGeneratorBuilder}; the old one
 * will still exist unchanged.  A {@code StringGeneratorBuilder} will always remain valid and reusable, even if {@code build}
 * has already been called.
 */
public abstract class StringGeneratorBuilder implements CoProduct2<StringGeneratorBuilder.Plain,
        StringGeneratorBuilder.Maybes, StringGeneratorBuilder> {
    private static Generator<String> applyStartDelimiter(Maybe<Generator<String>> startDelimiter,
                                                         Generator<String> gen) {
        return startDelimiter.match(__ -> gen, d -> Generators.generateString(d, gen));
    }

    private static Generator<String> applyEndDelimiter(Maybe<Generator<String>> endDelimiter,
                                                       Generator<String> gen) {
        return endDelimiter.match(__ -> gen, d -> Generators.generateString(gen, d));
    }

    static StringGeneratorBuilder builder() {
        return Plain.EMPTY;
    }

    /**
     * Adds a segment to the output that is to be generated by a {@link Generator}.
     *
     * @param gen a generator that generates a string that will concatenated to the final output
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder add(Generator<String> gen);

    /**
     * Adds a constant segment to the output.
     *
     * @param s a string that will be concatenated to the final output
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder add(String s);

    /**
     * Maybe adds a segment to the output that is to be generated by a {@link Generator}.
     *
     * @param gen a generator that generates a {@code Maybe<String>}, that will concatenated to the final output if {@code just}.
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder addMaybe(Generator<Maybe<String>> gen);

    /**
     * Maybe adds a constant segment to the output.
     *
     * @param s a {@code Maybe<String>}, that if {@code just}, will be concatenated to the final output
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder addMaybe(Maybe<String> s);

    /**
     * Adds zero or more segments to the output, each of which are generated by its own {@link Generator}.
     *
     * @param gens a sequence of zero or more string generators
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder addMany(Iterable<Generator<String>> gens);

    /**
     * Adds zero or more constant segments to the output.
     *
     * @param values a sequence of zero or more strings
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder addManyValues(Iterable<String> values);

    /**
     * Adds many segments to the output, each of which are generated by its own {@link Generator}.
     *
     * @param gens a sequence of zero or more {@code Generator<Maybe<String>>}s, which, when yielding
     *             {@code just}, concatenates the generated string to the output
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder addManyMaybe(Iterable<Generator<Maybe<String>>> gens);

    /**
     * Adds zero or more constant (maybe) segments to the output
     *
     * @param values a sequence of zero or more {@code Maybe<String>}s, which will be concatenated to the
     *               final output when {@code just}
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder addManyMaybeValues(Iterable<Maybe<String>> values);

    /**
     * Creates a new {@code StringGeneratorBuilder} that is the same as this one, with the separator to be generated
     * by the provided {@link Generator}.  This replaces any previous value for the separator.
     * <p>
     * The separator will be placed between segments in the final output.  The {@code Generator} for the separator will
     * be invoked <i>every time</i> a separator is called for in the output, not just once per output.
     *
     * @param newSeparator the {@code Generator} for separators
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder withSeparator(Generator<String> newSeparator);

    /**
     * Creates a new {@code StringGeneratorBuilder} that is the same as this one, with separator set to be
     * a constant value.  This replaces any previous value for the separator.
     * <p>
     * The separator will be placed between segments in the final output.
     *
     * @param newSeparator the separator value
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder withSeparator(String newSeparator);

    /**
     * Creates a new {@code StringGeneratorBuilder} that is the same as this one, with the start delimiter to be generated
     * by the provided {@link Generator}.  This replaces any previous value for the start delimiter.
     * <p>
     * The start delimiter is placed before the first segment in the final output (even if there are no segments).
     *
     * @param newDelimiter a {@code Generator} for the start delimiter
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder withStartDelimiter(Generator<String> newDelimiter);

    /**
     * Creates a new {@code StringGeneratorBuilder} that is the same as this one, with the start delimiter set to be
     * a constant value.  This replaces any previous value for the start delimiter.
     * <p>
     * The start delimiter is placed before the first segment in the final output (even if there are no segments).
     *
     * @param newDelimiter the start delimiter value
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder withStartDelimiter(String newDelimiter);

    /**
     * Creates a new {@code StringGeneratorBuilder} that is the same as this one, with the end delimiter to be generated
     * by the provided {@link Generator}.  This replaces any previous value for the end delimiter.
     * <p>
     * The end delimiter is placed after the last segment in the final output (even if there are no segments).
     *
     * @param newDelimiter a {@code Generator} for the end delimiter
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder withEndDelimiter(Generator<String> newDelimiter);

    /**
     * Creates a new {@code StringGeneratorBuilder} that is the same as this one, with the end delimiter set to be
     * a constant value.  This replaces any previous value for the end delimiter.
     * <p>
     * The end delimiter is placed after the last segment in the final output (even if there are no segments).
     *
     * @param newDelimiter the end delimiter value
     * @return a {@code StringGeneratorBuilder}
     */
    public abstract StringGeneratorBuilder withEndDelimiter(String newDelimiter);

    /**
     * Creates a {@link Generator} from this {@code StringGeneratorBuilder}.  The resulting {@code Generator} will,
     * each time it is invoked, generate all segments that were specified and concatenate them (optionally separated by the
     * separator and surrounded by the delimiters).
     * <p>
     * This {@code StringGeneratorBuilder} will remain valid after calling {@code build}, and can continue to be
     * built upon.
     *
     * @return a {@code Generator<String>}
     */
    public abstract Generator<String> build();

    static class Plain extends StringGeneratorBuilder {
        static StringGeneratorBuilder EMPTY = new Plain(Vector.empty(), nothing(), nothing(), nothing());

        private final ImmutableIterable<Generator<String>> components;
        private final Maybe<Generator<String>> separator;
        private final Maybe<Generator<String>> startDelimiter;
        private final Maybe<Generator<String>> endDelimiter;

        public Plain(ImmutableIterable<Generator<String>> components, Maybe<Generator<String>> separator, Maybe<Generator<String>> startDelimiter, Maybe<Generator<String>> endDelimiter) {
            this.components = components;
            this.separator = separator;
            this.startDelimiter = startDelimiter;
            this.endDelimiter = endDelimiter;
        }

        @Override
        public <R> R match(Fn1<? super Plain, ? extends R> aFn, Fn1<? super Maybes, ? extends R> bFn) {
            return aFn.apply(this);
        }

        Maybes toMaybes() {
            return new Maybes(components.fmap(Generator::just), separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder add(Generator<String> gen) {
            return new Plain(components.append(gen), separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder add(String s) {
            return null == s || s.equals("")
                    ? this
                    : add(constant(s));
        }

        public StringGeneratorBuilder addMaybe(Generator<Maybe<String>> gen) {
            return toMaybes().addMaybe(gen);
        }

        public StringGeneratorBuilder addMaybe(Maybe<String> s) {
            return s.match(__ -> this, this::add);
        }

        public StringGeneratorBuilder addMany(Iterable<Generator<String>> gens) {
            return new Plain(components.concat(Vector.copyFrom(gens)), separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder addManyValues(Iterable<String> values) {
            return new Plain(components.concat(Vector.copyFrom(values).fmap(Generators::constant)),
                    separator, startDelimiter, endDelimiter);

        }

        public StringGeneratorBuilder addManyMaybe(Iterable<Generator<Maybe<String>>> gens) {
            ImmutableVector<Generator<Maybe<String>>> gs = Vector.copyFrom(gens);
            if (gs.isEmpty()) {
                return this;
            } else {
                return this.toMaybes().addManyMaybe(gs);
            }
        }

        public StringGeneratorBuilder addManyMaybeValues(Iterable<Maybe<String>> values) {
            ImmutableVector<Maybe<String>> gs = Vector.copyFrom(values);
            if (gs.isEmpty()) {
                return this;
            } else {
                return this.toMaybes().addManyMaybeValues(gs);
            }
        }

        public StringGeneratorBuilder withSeparator(Generator<String> newSeparator) {
            return new Plain(components, just(newSeparator), startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder withSeparator(String newSeparator) {
            if (null == newSeparator || newSeparator.equals("")) {
                return separator.match(__ -> this,
                        __ -> new Plain(components, nothing(), startDelimiter, endDelimiter));
            } else {
                return withSeparator(constant(newSeparator));
            }
        }

        public StringGeneratorBuilder withStartDelimiter(Generator<String> newDelimiter) {
            return new Plain(components, separator, just(newDelimiter), endDelimiter);
        }

        public StringGeneratorBuilder withStartDelimiter(String newDelimiter) {
            if (null == newDelimiter || newDelimiter.equals("")) {
                return separator.match(__ -> this,
                        __ -> new Plain(components, separator, nothing(), endDelimiter));
            } else {
                return withStartDelimiter(constant(newDelimiter));
            }
        }

        public StringGeneratorBuilder withEndDelimiter(Generator<String> newDelimiter) {
            return new Plain(components, separator, startDelimiter, just(newDelimiter));
        }

        public StringGeneratorBuilder withEndDelimiter(String newDelimiter) {
            if (null == newDelimiter || newDelimiter.equals("")) {
                return separator.match(__ -> this,
                        __ -> new Plain(components, separator, startDelimiter, nothing()));
            } else {
                return withEndDelimiter(constant(newDelimiter));
            }
        }

        public Generator<String> build() {
            Generator<String> inner = separator.match(__ -> Strings.concatStrings(components),
                    sep -> Strings.concatStrings(sep, components));
            return StringGeneratorBuilder.applyStartDelimiter(startDelimiter,
                    StringGeneratorBuilder.applyEndDelimiter(endDelimiter, inner));
        }
    }

    static class Maybes extends StringGeneratorBuilder {
        private final Maybe<Generator<String>> separator;
        private final Maybe<Generator<String>> startDelimiter;
        private final Maybe<Generator<String>> endDelimiter;
        ImmutableIterable<Generator<Maybe<String>>> components;

        public Maybes(ImmutableIterable<Generator<Maybe<String>>> components, Maybe<Generator<String>> separator, Maybe<Generator<String>> startDelimiter, Maybe<Generator<String>> endDelimiter) {
            this.components = components;
            this.separator = separator;
            this.startDelimiter = startDelimiter;
            this.endDelimiter = endDelimiter;
        }

        @Override
        public <R> R match(Fn1<? super Plain, ? extends R> aFn, Fn1<? super Maybes, ? extends R> bFn) {
            return bFn.apply(this);
        }

        public StringGeneratorBuilder add(Generator<String> gen) {
            return addMaybe(gen.just());
        }

        public StringGeneratorBuilder add(String s) {
            return null == s || s.equals("")
                    ? this
                    : add(constant(s));
        }

        public StringGeneratorBuilder addMaybe(Generator<Maybe<String>> gen) {
            return new Maybes(components.append(gen), separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder addMaybe(Maybe<String> s) {
            return s.match(__ -> this, this::add);
        }

        public StringGeneratorBuilder addMany(Iterable<Generator<String>> gens) {
            return new Maybes(components.concat(Vector.copyFrom(gens).fmap(Generator::just)),
                    separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder addManyValues(Iterable<String> values) {
            return new Maybes(components.concat(Vector.copyFrom(values).fmap(s -> constant(s).just())),
                    separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder addManyMaybe(Iterable<Generator<Maybe<String>>> gens) {
            return new Maybes(components.concat(Vector.copyFrom(gens)),
                    separator, startDelimiter, endDelimiter);

        }

        public StringGeneratorBuilder addManyMaybeValues(Iterable<Maybe<String>> values) {
            return new Maybes(components.concat(Vector.copyFrom(values).fmap(Generators::constant)),
                    separator, startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder withSeparator(Generator<String> newSeparator) {
            return new Maybes(components, just(newSeparator), startDelimiter, endDelimiter);
        }

        public StringGeneratorBuilder withSeparator(String newSeparator) {
            if (null == newSeparator || newSeparator.equals("")) {
                return separator.match(__ -> this, __ -> new Maybes(components, nothing(), startDelimiter, endDelimiter));
            } else {
                return withSeparator(constant(newSeparator));
            }
        }

        public StringGeneratorBuilder withStartDelimiter(Generator<String> newDelimiter) {
            return new Maybes(components, separator, just(newDelimiter), endDelimiter);
        }

        public StringGeneratorBuilder withStartDelimiter(String newDelimiter) {
            if (null == newDelimiter || newDelimiter.equals("")) {
                return separator.match(__ -> this,
                        __ -> new Maybes(components, separator, nothing(), endDelimiter));
            } else {
                return withStartDelimiter(constant(newDelimiter));
            }
        }

        public StringGeneratorBuilder withEndDelimiter(Generator<String> newDelimiter) {
            return new Maybes(components, separator, startDelimiter, just(newDelimiter));
        }

        public StringGeneratorBuilder withEndDelimiter(String newDelimiter) {
            if (null == newDelimiter || newDelimiter.equals("")) {
                return separator.match(__ -> this,
                        __ -> new Maybes(components, separator, startDelimiter, nothing()));
            } else {
                return withEndDelimiter(constant(newDelimiter));
            }
        }

        public Generator<String> build() {
            Generator<String> inner = separator.match(__ -> Strings.concatMaybeStrings(components),
                    sep -> Strings.concatMaybeStrings(sep, components));
            return StringGeneratorBuilder.applyStartDelimiter(startDelimiter,
                    StringGeneratorBuilder.applyEndDelimiter(endDelimiter, inner));
        }
    }
}
