package software.crldev.elrondspringbootstarterreactive.domain.smartcontract;

import org.apache.logging.log4j.util.Strings;
import org.bouncycastle.util.encoders.Hex;
import software.crldev.elrondspringbootstarterreactive.error.exception.FunctionArgsException;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import static io.netty.util.internal.StringUtil.isNullOrEmpty;

/**
 * Value object for smart contract function arguments
 *
 * @author carlo_stanciu
 */
public final class FunctionArgs {
    private final List<String> value;

    private FunctionArgs() {
        this.value = Collections.emptyList();
    }

    private FunctionArgs(String... args) {
        verifyArgs(args);
        this.value = Stream.of(args).map(String::trim).map(String::getBytes).map(Hex::toHexString).collect(Collectors.toList());
    }

    /**
     * Creates a FunctionArgs objectt from String array
     *
     * @param values - string values
     * @return - an instance of FunctionArgs
     */
    public static FunctionArgs fromString(String... values) {
        return new FunctionArgs(values);
    }

    /**
     * Creates a FunctionArgs object with empty array of args
     *
     * @return - an instance of FunctionArgs
     */
    public static FunctionArgs empty() {
        return new FunctionArgs();
    }

    private void verifyArgs(String[] args) {
        if (args.length == 0 || hasEmptyOrNullArg(args)) throw new FunctionArgsException();
        Stream.of(args).findAny().map(a -> {
            if (isNullOrEmpty(a)) {
                throw new FunctionArgsException();
            }
            return a;
        });
    }

    private boolean hasEmptyOrNullArg(String[] args) {
        for (var arg : args) if (isNullOrEmpty(arg)) return true;
        return false;
    }

    /**
     * returns a String formed with the args in hex format
     * joined by "@" delimiter
     *
     * @return - args string hex with delimiter @
     */
    @Override
    public String toString() {
        return value.isEmpty() ? Strings.EMPTY : value.stream().map("@"::concat).collect(Collectors.joining());
    }

    public List<String> getValue() {
        return this.value;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == this) return true;
        if (!(o instanceof FunctionArgs)) return false;
        final FunctionArgs other = (FunctionArgs) o;
        final Object this$value = this.getValue();
        final Object other$value = other.getValue();
        if (this$value == null ? other$value != null : !this$value.equals(other$value)) return false;
        return true;
    }

    @Override
    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        final Object $value = this.getValue();
        result = result * PRIME + ($value == null ? 43 : $value.hashCode());
        return result;
    }
}
