package software.crldev.elrondspringbootstarterreactive.domain.smartcontract;

import lombok.NonNull;
import software.crldev.elrondspringbootstarterreactive.domain.account.Address;
import software.crldev.elrondspringbootstarterreactive.domain.common.Balance;
import software.crldev.elrondspringbootstarterreactive.domain.transaction.GasLimit;
import software.crldev.elrondspringbootstarterreactive.domain.transaction.PayloadData;

/**
 * Value object for transaction PayloadData
 *
 * @author carlo_stanciu
 */
public final class ScFunction {
    @NonNull
    private final Address smartContractAddress;
    @NonNull
    private final FunctionName functionName;
    @NonNull
    private final FunctionArgs args;
    private final Balance value;
    private final GasLimit gasLimit;

    /**
     * Retrieves an instance of PayloadData
     * containing the function name and arguments in the required format
     *
     * @return - an instance of PayloadData
     */
    public PayloadData getPayloadData() {
        return PayloadData.fromString(this.functionName + this.args.toString());
    }

    ScFunction(@NonNull final Address smartContractAddress, @NonNull final FunctionName functionName, @NonNull final FunctionArgs args, final Balance value, final GasLimit gasLimit) {
        if (smartContractAddress == null) {
            throw new NullPointerException("smartContractAddress is marked non-null but is null");
        }
        if (functionName == null) {
            throw new NullPointerException("functionName is marked non-null but is null");
        }
        if (args == null) {
            throw new NullPointerException("args is marked non-null but is null");
        }
        this.smartContractAddress = smartContractAddress;
        this.functionName = functionName;
        this.args = args;
        this.value = value;
        this.gasLimit = gasLimit;
    }


    public static class ScFunctionBuilder {
        private Address smartContractAddress;
        private FunctionName functionName;
        private FunctionArgs args;
        private Balance value;
        private GasLimit gasLimit;

        ScFunctionBuilder() {
        }

        /**
         * @return {@code this}.
         */
        public ScFunction.ScFunctionBuilder smartContractAddress(@NonNull final Address smartContractAddress) {
            if (smartContractAddress == null) {
                throw new NullPointerException("smartContractAddress is marked non-null but is null");
            }
            this.smartContractAddress = smartContractAddress;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public ScFunction.ScFunctionBuilder functionName(@NonNull final FunctionName functionName) {
            if (functionName == null) {
                throw new NullPointerException("functionName is marked non-null but is null");
            }
            this.functionName = functionName;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public ScFunction.ScFunctionBuilder args(@NonNull final FunctionArgs args) {
            if (args == null) {
                throw new NullPointerException("args is marked non-null but is null");
            }
            this.args = args;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public ScFunction.ScFunctionBuilder value(final Balance value) {
            this.value = value;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public ScFunction.ScFunctionBuilder gasLimit(final GasLimit gasLimit) {
            this.gasLimit = gasLimit;
            return this;
        }

        public ScFunction build() {
            return new ScFunction(this.smartContractAddress, this.functionName, this.args, this.value, this.gasLimit);
        }

        @Override
        public String toString() {
            return "ScFunction.ScFunctionBuilder(smartContractAddress=" + this.smartContractAddress + ", functionName=" + this.functionName + ", args=" + this.args + ", value=" + this.value + ", gasLimit=" + this.gasLimit + ")";
        }
    }

    public static ScFunction.ScFunctionBuilder builder() {
        return new ScFunction.ScFunctionBuilder();
    }

    @NonNull
    public Address getSmartContractAddress() {
        return this.smartContractAddress;
    }

    @NonNull
    public FunctionName getFunctionName() {
        return this.functionName;
    }

    @NonNull
    public FunctionArgs getArgs() {
        return this.args;
    }

    public Balance getValue() {
        return this.value;
    }

    public GasLimit getGasLimit() {
        return this.gasLimit;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == this) return true;
        if (!(o instanceof ScFunction)) return false;
        final ScFunction other = (ScFunction) o;
        final Object this$smartContractAddress = this.getSmartContractAddress();
        final Object other$smartContractAddress = other.getSmartContractAddress();
        if (this$smartContractAddress == null ? other$smartContractAddress != null : !this$smartContractAddress.equals(other$smartContractAddress)) return false;
        final Object this$functionName = this.getFunctionName();
        final Object other$functionName = other.getFunctionName();
        if (this$functionName == null ? other$functionName != null : !this$functionName.equals(other$functionName)) return false;
        final Object this$args = this.getArgs();
        final Object other$args = other.getArgs();
        if (this$args == null ? other$args != null : !this$args.equals(other$args)) return false;
        final Object this$value = this.getValue();
        final Object other$value = other.getValue();
        if (this$value == null ? other$value != null : !this$value.equals(other$value)) return false;
        final Object this$gasLimit = this.getGasLimit();
        final Object other$gasLimit = other.getGasLimit();
        if (this$gasLimit == null ? other$gasLimit != null : !this$gasLimit.equals(other$gasLimit)) return false;
        return true;
    }

    @Override
    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        final Object $smartContractAddress = this.getSmartContractAddress();
        result = result * PRIME + ($smartContractAddress == null ? 43 : $smartContractAddress.hashCode());
        final Object $functionName = this.getFunctionName();
        result = result * PRIME + ($functionName == null ? 43 : $functionName.hashCode());
        final Object $args = this.getArgs();
        result = result * PRIME + ($args == null ? 43 : $args.hashCode());
        final Object $value = this.getValue();
        result = result * PRIME + ($value == null ? 43 : $value.hashCode());
        final Object $gasLimit = this.getGasLimit();
        result = result * PRIME + ($gasLimit == null ? 43 : $gasLimit.hashCode());
        return result;
    }

    @Override
    public String toString() {
        return "ScFunction(smartContractAddress=" + this.getSmartContractAddress() + ", functionName=" + this.getFunctionName() + ", args=" + this.getArgs() + ", value=" + this.getValue() + ", gasLimit=" + this.getGasLimit() + ")";
    }
}
