package software.crldev.elrondspringbootstarterreactive.domain.transaction;

import software.crldev.elrondspringbootstarterreactive.config.ErdNetworkConfigSupplier;
import software.crldev.elrondspringbootstarterreactive.config.JsonMapper;
import software.crldev.elrondspringbootstarterreactive.domain.account.Address;
import software.crldev.elrondspringbootstarterreactive.domain.common.Balance;
import software.crldev.elrondspringbootstarterreactive.domain.common.Nonce;
import software.crldev.elrondspringbootstarterreactive.interactor.transaction.SendableTransaction;
import com.fasterxml.jackson.core.JsonProcessingException;
import static software.crldev.elrondspringbootstarterreactive.util.GasUtils.computeGasCost;

/**
 * Domain object for Transaction representation, which is Signable
 * Contains all the required fields to validate & construct a transaction
 *
 * @author carlo_stanciu
 */
public class Transaction implements Signable {
    private Nonce nonce = Nonce.fromLong(0L);
    private ChainID chainID = ChainID.fromString(ErdNetworkConfigSupplier.config.getChainId());
    private Balance value = Balance.zero();
    private Address sender = Address.zero();
    private Address receiver = Address.zero();
    private GasPrice gasPrice = GasPrice.fromNumber(ErdNetworkConfigSupplier.config.getMinGasPrice());
    private GasLimit gasLimit = GasLimit.fromNumber(ErdNetworkConfigSupplier.config.getMinGasLimit());
    private PayloadData payloadData = PayloadData.empty();
    private TransactionVersion version = TransactionVersion.withDefaultVersion();
    private Signature signature = Signature.empty();
    private Hash hash = Hash.empty();
    private TransactionStatus status = TransactionStatus.UNKNOWN;

    @Override
    public byte[] serializeForSigning() throws JsonProcessingException {
        return JsonMapper.serializeToJsonBuffer(toSendable());
    }

    @Override
    public void applySignature(Signature signature) {
        setSignature(signature);
    }

    /**
     * Method use to create a Sendable representation of the transaction
     * used for submitting send/sendMultiple/simulate requests to the TransactionInteractor
     *
     * @return - instance of SendableTransaction
     */
    public SendableTransaction toSendable() {
        return generateSender(false);
    }

    /**
     * Method use to create a Sendable representation of the transaction
     * used for submitting estimate request to the TransactionInteractor
     *
     * @return - instance of SendableTransaction
     */
    public SendableTransaction toSendableForEstimation() {
        return generateSender(true);
    }

    /**
     * Setter
     * Used for setting PayloadData
     * and also computing GasCost for the data
     *
     * @param payloadData - data input value
     */
    public void setPayloadData(PayloadData payloadData) {
        this.payloadData = payloadData;
        this.gasLimit = GasLimit.fromNumber(computeGasCost(payloadData));
    }

    private SendableTransaction generateSender(boolean isEstimation) {
        var sendableBuilder = SendableTransaction.builder().chainId(chainID.getValue()).nonce(nonce.getValue()).value(value.toString()).receiver(receiver.getBech32()).sender(sender.getBech32()).data(payloadData.isEmpty() ? null : payloadData.encoded()).signature(signature.isEmpty() ? null : signature.getHex()).version(version.getValue());
        if (!isEstimation) {
            sendableBuilder.gasLimit(gasLimit.getValue());
            sendableBuilder.gasPrice(gasPrice.getValue());
        }
        return sendableBuilder.build();
    }

    public Transaction() {
    }

    public Nonce getNonce() {
        return this.nonce;
    }

    public ChainID getChainID() {
        return this.chainID;
    }

    public Balance getValue() {
        return this.value;
    }

    public Address getSender() {
        return this.sender;
    }

    public Address getReceiver() {
        return this.receiver;
    }

    public GasPrice getGasPrice() {
        return this.gasPrice;
    }

    public GasLimit getGasLimit() {
        return this.gasLimit;
    }

    public PayloadData getPayloadData() {
        return this.payloadData;
    }

    public TransactionVersion getVersion() {
        return this.version;
    }

    public Signature getSignature() {
        return this.signature;
    }

    public Hash getHash() {
        return this.hash;
    }

    public TransactionStatus getStatus() {
        return this.status;
    }

    public void setNonce(final Nonce nonce) {
        this.nonce = nonce;
    }

    public void setChainID(final ChainID chainID) {
        this.chainID = chainID;
    }

    public void setValue(final Balance value) {
        this.value = value;
    }

    public void setSender(final Address sender) {
        this.sender = sender;
    }

    public void setReceiver(final Address receiver) {
        this.receiver = receiver;
    }

    public void setGasPrice(final GasPrice gasPrice) {
        this.gasPrice = gasPrice;
    }

    public void setGasLimit(final GasLimit gasLimit) {
        this.gasLimit = gasLimit;
    }

    public void setVersion(final TransactionVersion version) {
        this.version = version;
    }

    public void setSignature(final Signature signature) {
        this.signature = signature;
    }

    public void setHash(final Hash hash) {
        this.hash = hash;
    }

    public void setStatus(final TransactionStatus status) {
        this.status = status;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == this) return true;
        if (!(o instanceof Transaction)) return false;
        final Transaction other = (Transaction) o;
        if (!other.canEqual((Object) this)) return false;
        final Object this$nonce = this.getNonce();
        final Object other$nonce = other.getNonce();
        if (this$nonce == null ? other$nonce != null : !this$nonce.equals(other$nonce)) return false;
        final Object this$chainID = this.getChainID();
        final Object other$chainID = other.getChainID();
        if (this$chainID == null ? other$chainID != null : !this$chainID.equals(other$chainID)) return false;
        final Object this$value = this.getValue();
        final Object other$value = other.getValue();
        if (this$value == null ? other$value != null : !this$value.equals(other$value)) return false;
        final Object this$sender = this.getSender();
        final Object other$sender = other.getSender();
        if (this$sender == null ? other$sender != null : !this$sender.equals(other$sender)) return false;
        final Object this$receiver = this.getReceiver();
        final Object other$receiver = other.getReceiver();
        if (this$receiver == null ? other$receiver != null : !this$receiver.equals(other$receiver)) return false;
        final Object this$gasPrice = this.getGasPrice();
        final Object other$gasPrice = other.getGasPrice();
        if (this$gasPrice == null ? other$gasPrice != null : !this$gasPrice.equals(other$gasPrice)) return false;
        final Object this$gasLimit = this.getGasLimit();
        final Object other$gasLimit = other.getGasLimit();
        if (this$gasLimit == null ? other$gasLimit != null : !this$gasLimit.equals(other$gasLimit)) return false;
        final Object this$payloadData = this.getPayloadData();
        final Object other$payloadData = other.getPayloadData();
        if (this$payloadData == null ? other$payloadData != null : !this$payloadData.equals(other$payloadData)) return false;
        final Object this$version = this.getVersion();
        final Object other$version = other.getVersion();
        if (this$version == null ? other$version != null : !this$version.equals(other$version)) return false;
        final Object this$signature = this.getSignature();
        final Object other$signature = other.getSignature();
        if (this$signature == null ? other$signature != null : !this$signature.equals(other$signature)) return false;
        final Object this$hash = this.getHash();
        final Object other$hash = other.getHash();
        if (this$hash == null ? other$hash != null : !this$hash.equals(other$hash)) return false;
        final Object this$status = this.getStatus();
        final Object other$status = other.getStatus();
        if (this$status == null ? other$status != null : !this$status.equals(other$status)) return false;
        return true;
    }

    protected boolean canEqual(final Object other) {
        return other instanceof Transaction;
    }

    @Override
    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        final Object $nonce = this.getNonce();
        result = result * PRIME + ($nonce == null ? 43 : $nonce.hashCode());
        final Object $chainID = this.getChainID();
        result = result * PRIME + ($chainID == null ? 43 : $chainID.hashCode());
        final Object $value = this.getValue();
        result = result * PRIME + ($value == null ? 43 : $value.hashCode());
        final Object $sender = this.getSender();
        result = result * PRIME + ($sender == null ? 43 : $sender.hashCode());
        final Object $receiver = this.getReceiver();
        result = result * PRIME + ($receiver == null ? 43 : $receiver.hashCode());
        final Object $gasPrice = this.getGasPrice();
        result = result * PRIME + ($gasPrice == null ? 43 : $gasPrice.hashCode());
        final Object $gasLimit = this.getGasLimit();
        result = result * PRIME + ($gasLimit == null ? 43 : $gasLimit.hashCode());
        final Object $payloadData = this.getPayloadData();
        result = result * PRIME + ($payloadData == null ? 43 : $payloadData.hashCode());
        final Object $version = this.getVersion();
        result = result * PRIME + ($version == null ? 43 : $version.hashCode());
        final Object $signature = this.getSignature();
        result = result * PRIME + ($signature == null ? 43 : $signature.hashCode());
        final Object $hash = this.getHash();
        result = result * PRIME + ($hash == null ? 43 : $hash.hashCode());
        final Object $status = this.getStatus();
        result = result * PRIME + ($status == null ? 43 : $status.hashCode());
        return result;
    }

    @Override
    public String toString() {
        return "Transaction(nonce=" + this.getNonce() + ", chainID=" + this.getChainID() + ", value=" + this.getValue() + ", sender=" + this.getSender() + ", receiver=" + this.getReceiver() + ", gasPrice=" + this.getGasPrice() + ", gasLimit=" + this.getGasLimit() + ", payloadData=" + this.getPayloadData() + ", version=" + this.getVersion() + ", signature=" + this.getSignature() + ", hash=" + this.getHash() + ", status=" + this.getStatus() + ")";
    }
}
