package com.craftman.cardform;

import android.support.annotation.Nullable;

/**
 * Utility class for common text-related operations on Stripe data coming from the server.
 */
public class StripeTextUtils {

    /**
     * Check to see if the input number has any of the given prefixes.
     *
     * @param number   the number to test
     * @param prefixes the prefixes to test against
     * @return {@code true} if number begins with any of the input prefixes
     */
    public static boolean hasAnyPrefix(String number, String... prefixes) {
        if (number == null) {
            return false;
        }

        for (String prefix : prefixes) {
            if (number.startsWith(prefix)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check to see whether the input string is a whole, positive number.
     *
     * @param value the input string to test
     * @return {@code true} if the input value consists entirely of integers
     */
    public static boolean isWholePositiveNumber(String value) {
        if (value == null) {
            return false;
        }

        // Refraining from using android's TextUtils in order to avoid
        // depending on another package.
        for (int i = 0; i < value.length(); i++) {
            if (!Character.isDigit(value.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * Swap {@code null} for blank text values.
     *
     * @param value an input string that may or may not be entirely whitespace
     * @return {@code null} if the string is entirely whitespace, or the original value if not
     */
    public static String nullIfBlank(String value) {
        if (isBlank(value)) {
            return null;
        }
        return value;
    }

    /**
     * A checker for whether or not the input value is entirely whitespace. This is slightly more
     * aggressive than the android TextUtils#isEmpty method, which only returns true for
     * {@code null} or {@code ""}.
     *
     * @param value a possibly blank input string value
     * @return {@code true} if and only if the value is all whitespace, {@code null}, or empty
     */
    public static boolean isBlank(String value) {
        return value == null || value.trim().length() == 0;
    }


    @Nullable
    @CreditCard.CardBrand
    public static String asCardBrand(@Nullable String possibleCardType) {
        if (isBlank(possibleCardType)) {
            return null;
        }

        if (CreditCard.AMERICAN_EXPRESS.equalsIgnoreCase(possibleCardType)) {
            return CreditCard.AMERICAN_EXPRESS;
        } else if (CreditCard.MASTERCARD.equalsIgnoreCase(possibleCardType)) {
            return CreditCard.MASTERCARD;
        } else if (CreditCard.DINERS_CLUB.equalsIgnoreCase(possibleCardType)) {
            return CreditCard.DINERS_CLUB;
        } else if (CreditCard.DISCOVER.equalsIgnoreCase(possibleCardType)) {
            return CreditCard.DISCOVER;
        } else if (CreditCard.JCB.equalsIgnoreCase(possibleCardType)) {
            return CreditCard.JCB;
        } else if (CreditCard.VISA.equalsIgnoreCase(possibleCardType)) {
            return CreditCard.VISA;
        } else {
            return CreditCard.UNKNOWN;
        }
    }


    @Nullable
    @CreditCard.FundingType
    public static String asFundingType(@Nullable String possibleFundingType) {
        if (isBlank(possibleFundingType)) {
            return null;
        }

        if (CreditCard.FUNDING_CREDIT.equalsIgnoreCase(possibleFundingType)) {
            return CreditCard.FUNDING_CREDIT;
        } else if (CreditCard.FUNDING_DEBIT.equalsIgnoreCase(possibleFundingType)) {
            return CreditCard.FUNDING_DEBIT;
        } else if (CreditCard.FUNDING_PREPAID.equalsIgnoreCase(possibleFundingType)) {
            return CreditCard.FUNDING_PREPAID;
        } else {
            return CreditCard.FUNDING_UNKNOWN;
        }
    }

    @Nullable
    @Token.TokenType
    public static String asTokenType(@Nullable String possibleTokenType) {
        if (Token.TYPE_CARD.equals(possibleTokenType)) {
            return Token.TYPE_CARD;
        }
        return null;
    }
}
