package smartthings.ratpack.brave;

import brave.sampler.BoundarySampler;
import brave.sampler.RateLimitingSampler;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSCredentialsProviderChain;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceAsyncClientBuilder;
import ratpack.zipkin.ServerTracingModule;
import zipkin2.Span;
import zipkin2.reporter.AsyncReporter;
import zipkin2.reporter.Reporter;
import zipkin2.reporter.sqs.SQSSender;

/**
 * Configuration adapter for typical SmartThings Zipkin configuration.
 */
public class ServerTracingConfig {

	private String serviceName;
	private String reporter;
	private float sampleRate;
	private String samplerKind = "BOUNDARY"; // default to BOUNDARY for backwards compatibility
	private String queueUrl;
	private String awsAccessKey;
	private String awsSecretAccessKey;
	private String awsRoleArn;

	public String getServiceName() {
		return this.serviceName;
	}

	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}

	public String getReporter() {
		return this.reporter;
	}

	public void setReporter(String reporter) {
		this.reporter = reporter;
	}

	public float getSampleRate() {
		return this.sampleRate;
	}

	public void setSampleRate(float sampleRate) {
		this.sampleRate = sampleRate;
	}

	public String getSamplerKind() {
		return this.samplerKind;
	}

	public void setSamplerKind(String samplerKind) {
		this.samplerKind = samplerKind;
	}

	public String getQueueUrl() {
		return this.queueUrl;
	}

	public void setQueueUrl(String queueUrl) {
		this.queueUrl = queueUrl;
	}

	public String getAwsAccessKey() {
		return this.awsAccessKey;
	}

	public void setAwsAccessKey(String awsAccessKey) {
		this.awsAccessKey = awsAccessKey;
	}

	public String getAwsSecretAccessKey() {
		return this.awsSecretAccessKey;
	}

	public void setAwsSecretAccessKey(String awsSecretAccessKey) {
		this.awsSecretAccessKey = awsSecretAccessKey;
	}

	public String getAwsRoleArn() {
		return this.awsRoleArn;
	}

	public void setAwsRoleArn(String awsRoleArn) {
		this.awsRoleArn = awsRoleArn;
	}

	public ServerTracingModule.Config toConfig() {
		return new ServerTracingModule.Config()
				.serviceName(serviceName)
				.spanReporterV2(getSpanReporter())
				.sampler((samplerKind.equals("RATE_LIMITING")) ?
						RateLimitingSampler.create(Math.round(sampleRate)) :
						BoundarySampler.create(sampleRate));
	}

	private Reporter<Span> getSpanReporter() {
		if ("CONSOLE".equals(reporter)) {
			return Reporter.CONSOLE;
		} else if ("SQS".equals(reporter)) {
			return AsyncReporter.builder(getSqsSender()).build();
		} else {
			return Reporter.NOOP;
		}
	}

	private SQSSender getSqsSender() {

		AWSCredentialsProvider provider;

		if (awsAccessKey != null && awsSecretAccessKey != null) {
			provider = new AWSCredentialsProviderChain(
					new AWSStaticCredentialsProvider(
							new BasicAWSCredentials(awsAccessKey, awsSecretAccessKey))
			);
		} else {
			provider = new DefaultAWSCredentialsProviderChain();
		}

		if (awsRoleArn != null) {
			provider = new STSAssumeRoleSessionCredentialsProvider.Builder(awsRoleArn, "zipkin")
					.withStsClient(
							AWSSecurityTokenServiceAsyncClientBuilder
									.standard()
									.withCredentials(provider).build()).build();
		}

		return SQSSender.newBuilder().queueUrl(queueUrl).credentialsProvider(provider).build();
	}

}
