package smartrics.iotics.connectors.twins;

import com.google.common.util.concurrent.ListenableFuture;
import com.iotics.api.UpsertTwinRequest;
import com.iotics.api.UpsertTwinResponse;

/**
 * Interface that extends the functionalities of Maker and Mappable to support upsert operations.
 * It leverages the mapping capabilities to generate request objects for upserting twins based on the current state
 * of the object. This interface is useful in scenarios where twin data needs to be created or updated dynamically
 * based on an evolving state or in response to external events.
 */
public interface MappableMaker extends Maker, Mappable {

    /**
     * Performs an upsert operation for a twin, using a request generated by the {@link Mapper} associated with this object.
     * This method fetches an {@link UpsertTwinRequest} from the {@link Mapper} and sends it to the twin API, handling the response
     * asynchronously.
     *
     * @return a {@link ListenableFuture} containing the response from the upsert operation, allowing for asynchronous processing.
     */
    default ListenableFuture<UpsertTwinResponse> upsert() {
        UpsertTwinRequest upsertRequest = getMapper().getUpsertTwinRequest();
        return ioticsApi().twinAPIFuture().upsertTwin(upsertRequest);
    }

}