/*
    Copyright 2019-2020 eMobilify GmbH

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/
package shareandcharge.openchargingnetwork.notary

import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.annotation.JsonProperty

/**
 * OCPI request headers incl. correlation id and routing headers. Excludes X-Request-ID as this is always replaced.
 */
data class OcpiRequestHeaders(@JsonProperty("x-correlation-id")        var correlationId: String,
                              @JsonProperty("ocpi-from-country-code")  var fromCountryCode: String,
                              @JsonProperty("ocpi-from-party-id")      var fromPartyId: String,
                              @JsonProperty("ocpi-to-country-code")    var toCountryCode: String,
                              @JsonProperty("ocpi-to-party-id")        var toPartyId: String)

/**
 * OCPI response headers, e.g. pagination/rest headers
 */
data class OcpiResponseHeaders(@JsonProperty("x-limit")        var limit: String,
                               @JsonProperty("x-total-count")  var totalCount: String,
                               @JsonProperty("link")           var link: String,
                               @JsonProperty("location")       var location: String)

/**
 * OCPI unique id and routing headers. 
 */
data class OcpiHeaders(var request: OcpiRequestHeaders? = null,
                       var response: OcpiResponseHeaders? = null)

/**
 * OcpiRequest provides the values which should be signed.
 * @property headers OCPI request headers
 * @property params optional url-encoded parameters
 * @property body generic body for the appropriate OCPI request
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
data class OcpiRequest<T>(var headers: OcpiHeaders,
                          var params: Map<String, Any>? = null,
                          var body: T? = null)

/**
 * Result of the verify method.
 * @property isValid passed verification
 * @property error optional error message if verification not successful
 */
data class VerifyResult(val isValid: Boolean, val error: String? = null)

/**
 * Result of the Rewrite verify method.
 * @property isValid passed verification
 * @property error optional error message if verification not successful
 * @property previousValues optional re-built OcpiRequest if verification successful
 */
data class RewriteVerifyResult(val isValid: Boolean,
                               val error: String? = null,
                               val previousValues: OcpiRequest<*>? = null)