package shareandcharge.openchargingnetwork.notary

import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.annotation.JsonProperty


/**
 * OCPI unique id and routing headers. Excludes X-Request-ID as this is always replaced.
 */
data class OcpiHeaders(@JsonProperty("x-correlation-id")        var correlationId: String,
                       @JsonProperty("ocpi-from-country-code")  var fromCountryCode: String,
                       @JsonProperty("ocpi-from-party-id")      var fromPartyId: String,
                       @JsonProperty("ocpi-to-country-code")    var toCountryCode: String,
                       @JsonProperty("ocpi-to-party-id")        var toPartyId: String)

/**
 * OCPI url-encoded parameters for paginated requests OR particular token module requests.
 */
data class OcpiUrlEncodedParameters(@JsonProperty("country_code")   var countryCode: String? = null,
                                    @JsonProperty("party_id")       var partyId: String? = null,
                                    @JsonProperty("token_uid")      var tokenUid: String? = null,
                                    var type: String? = null,
                                    var offset: String? = null,
                                    var limit: String? = null,
                                    @JsonProperty("date_from")      var dateFrom: String? = null,
                                    @JsonProperty("date_to")        var date_to: String? = null)

/**
 * OcpiRequest provides the values which should be signed.
 * @property headers OCPI request headers
 * @property params optional url-encoded parameters
 * @property body generic body for the appropriate OCPI request
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
data class OcpiRequest<T>(var headers: OcpiHeaders,
                          var params: OcpiUrlEncodedParameters? = null,
                          var body: T? = null)

/**
 * Result of the verify method.
 * @property isValid passed verification
 * @property error optional error message if verification not successful
 */
data class VerifyResult(val isValid: Boolean, val error: String? = null)

/**
 * Result of the Rewrite verify method.
 * @property isValid passed verification
 * @property error optional error message if verification not successful
 * @property previousValues optional re-built OcpiRequest if verification successful
 */
data class RewriteVerifyResult(val isValid: Boolean,
                               val error: String? = null,
                               val previousValues: OcpiRequest<*>? = null)