/*
 * Copyright (c) 2021 the original author or authors.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

package science.aist.machinelearning.analytics.graph.nodes;


import org.neo4j.ogm.annotation.Id;
import org.neo4j.ogm.annotation.NodeEntity;
import org.neo4j.ogm.annotation.Relationship;
import science.aist.machinelearning.core.Problem;
import science.aist.machinelearning.core.Solution;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Base node that is the parent of an execution
 *
 * @author Oliver Krauss
 * @since 1.0
 */
@NodeEntity
public class AnalyticsNode {

    /**
     * Links to all steps taken during the run
     */
    @Relationship(type = "EXECUTED", direction = Relationship.OUTGOING)
    Set<StepNode> steps = new HashSet<>();
    /**
     * Problem statement the algorithm was supposed to solve
     */
    @Relationship(type = "PROBLEM", direction = Relationship.OUTGOING)
    Problem problem;
    /**
     * Solution the algorithm produced
     */
    @Relationship(type = "SOLUTION", direction = Relationship.OUTGOING)
    Solution solution;
    /**
     * Id generated by database
     */
    @Id
    private Long id;
    /**
     * Human readable title of Analytics run
     */
    private String title;
    /**
     * Parameters how an algorithm was configured, dependent on algorithm used during run
     */
    private Map<String, Object> parameters = new HashMap<>();

    public AnalyticsNode() {
    }

    public AnalyticsNode(String title) {
        this.title = title;
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public Map<String, Object> getParameters() {
        return parameters;
    }

    public void setParameters(Map<String, Object> parameters) {
        this.parameters = parameters;
    }

    public void addParameter(String name, Object value) {
        parameters.put(name, value);
    }

    public Set<StepNode> getSteps() {
        return steps;
    }

    public void setSteps(Set<StepNode> steps) {
        this.steps = steps;
    }

    public void addStep(StepNode step) {
        this.steps.add(step);
    }

    public Problem getProblem() {
        return problem;
    }

    public void setProblem(Problem problem) {
        this.problem = problem;
    }

    public Solution getSolution() {
        return solution;
    }

    public void setSolution(Solution solution) {
        this.solution = solution;
    }
}
