package pw.prok.kdiff.diff.visitors;

import pw.prok.kdiff.Chunk;
import pw.prok.kdiff.Pair;
import pw.prok.kdiff.Patch;
import pw.prok.kdiff.PatchResult;
import pw.prok.kdiff.delta.Delta;
import pw.prok.kdiff.diff.DiffException;
import pw.prok.kdiff.diff.DiffRow;
import pw.prok.kdiff.diff.DiffVisitor;

import java.util.*;

public class MappedVisitor<T, C> implements DiffVisitor<T, C> {
    private final Map<Pair<String, String>, Patch<T, PatchResult<T>>> map;

    public MappedVisitor() {
        this(new HashMap<Pair<String, String>, Patch<T, PatchResult<T>>>());
    }

    public MappedVisitor(Map<Pair<String, String>, Patch<T, PatchResult<T>>> map) {
        this.map = map;
    }

    public Map<Pair<String, String>, Patch<T, PatchResult<T>>> getMap() {
        return map;
    }

    @Override
    public boolean visitComment(C comment) {
        return true;
    }

    protected String originalFilename;
    protected Date originalDate;

    @Override
    public boolean visitOriginal(String original, Date date) throws DiffException {
        originalFilename = original;
        originalDate = date;
        return true;
    }

    protected String revisedFilename;
    protected Date revisedDate;

    @Override
    public boolean visitRevised(String revised, Date date) throws DiffException {
        revisedFilename = revised;
        revisedDate = date;
        return true;
    }

    @Override
    public boolean visitChunkStart(int originalPosition, int originalCount, int revisedPosition, int revisedCount) {
        return true;
    }

    final List<T> oldLines = new ArrayList<>();
    final List<T> newLines = new ArrayList<>();

    @Override
    public boolean visitChunkEnd(int originalPosition, int originalCount, int revisedPosition, int revisedCount) {
        final Chunk<T> original = new Chunk<>(originalPosition - 1, new ArrayList<>(oldLines));
        final Chunk<T> revised = new Chunk<>(revisedPosition - 1, new ArrayList<>(newLines));
        oldLines.clear();
        newLines.clear();
        final Pair<String, String> key = Pair.immutable(originalFilename, revisedFilename);
        Patch<T, PatchResult<T>> patch = map.get(key);
        if (patch == null) {
            map.put(key, patch = new Patch<>());
            patch.setOriginal(originalFilename);
            patch.setOriginalDate(originalDate);
            patch.setRevised(revisedFilename);
            patch.setRevisedDate(revisedDate);
        }
        patch.addDelta(new Delta<>(original, revised));
        return true;
    }

    @Override
    public boolean visitLine(DiffRow.Tag tag, T line) {
        switch (tag) {
            case INSERT:
                newLines.add(line);
                return true;
            case DELETE:
                oldLines.add(line);
                return true;
            default:
                oldLines.add(line);
                newLines.add(line);
                return true;
        }
    }
}
