package pw.prok.kdiff.diff.parsers;

import pw.prok.kdiff.diff.DiffException;
import pw.prok.kdiff.diff.DiffParser;
import pw.prok.kdiff.diff.DiffRow;
import pw.prok.kdiff.diff.DiffVisitor;

import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.ParseException;
import java.text.ParsePosition;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UnifedDiffParser<Visitor extends DiffVisitor<String, String>> extends DiffParser<String, String, Visitor> {
    public static final UnifedDiffParser<DiffVisitor<String, String>> INSTANCE = new UnifedDiffParser<>();
    public static class UnifedDiffDateFormat extends DateFormat {
        public static final UnifedDiffDateFormat INSTANCE = new UnifedDiffDateFormat();
        private static final Pattern PATTERN = Pattern.compile("^(?<year>\\d{4})-(?<month>\\d{2})-(?<day>\\d{2}) (?<hour>\\d{2}):(?<minute>\\d{2}):(?<second>\\d{2}).(?<millisecond>\\d{9}) (?<tzsign>\\+|-)(?<tzhour>\\d{2})(?<tzminute>\\d{2})$");
        private static final String FORMAT = "%04d-%02d-%02d %02d:%02d:%02d.%09d %s%02d%02d";

        @Override
        public StringBuffer format(Date date, StringBuffer buffer, FieldPosition position) {
            final Calendar calendar = Calendar.getInstance(Locale.US);
            calendar.setTime(date);
            final int year = calendar.get(Calendar.YEAR);
            final int month = calendar.get(Calendar.MONTH);
            final int day = calendar.get(Calendar.DAY_OF_MONTH);
            final int hour = calendar.get(Calendar.HOUR);
            final int minute = calendar.get(Calendar.MINUTE);
            final int second = calendar.get(Calendar.SECOND);
            final int millisecond = calendar.get(Calendar.MILLISECOND) * 1000000;
            final int offset = calendar.getTimeZone().getRawOffset();
            final String tzsign = offset >= 0 ? "+" : "-";
            final int tzhour = offset / 3600000;
            final int tzminute = offset % 3600000;
            buffer.append(String.format(FORMAT, year, month, day, hour, minute, second, millisecond, tzsign, tzhour, tzminute));
            return buffer;
        }

        @Override
        public Date parse(String s, ParsePosition position) {
            final Matcher matcher = PATTERN.matcher(s.subSequence(position.getIndex(), s.length()));
            if (!matcher.matches())
                return null;
            position.setIndex(position.getIndex() + matcher.end());
            final Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC" + matcher.group("tzsign") + matcher.group("tzhour") + ":" + matcher.group("tzminute")), Locale.US);
            calendar.set(Calendar.YEAR, Integer.parseInt(matcher.group("year")));
            calendar.set(Calendar.MONTH, Integer.parseInt(matcher.group("month")));
            calendar.set(Calendar.DAY_OF_MONTH, Integer.parseInt(matcher.group("day")));
            calendar.set(Calendar.HOUR, Integer.parseInt(matcher.group("hour")));
            calendar.set(Calendar.MINUTE, Integer.parseInt(matcher.group("minute")));
            calendar.set(Calendar.SECOND, Integer.parseInt(matcher.group("second")));
            calendar.set(Calendar.MILLISECOND, Integer.parseInt(matcher.group("millisecond")) / 1000000);
            return calendar.getTime();
        }
    }

    private static final Pattern CHUNK_PATTERN = Pattern.compile("^@@\\s*-(?<origpos>\\d+)(\\s*,\\s*(?<origlen>\\d+))?\\s+\\+(?<revipos>\\d+)(\\s*,\\s*(?<revilen>\\d+))?\\s*@@$");

    @Override
    public void parse(List<String> elements, Visitor visitor) throws DiffException {
        final Iterator<String> iterator = elements.iterator();
        boolean next = true;
        while (iterator.hasNext() && next) {
            String element = iterator.next();
            if (element.startsWith("---")) {
                next = splitFilename(element, visitor, true);
                continue;
            }
            if (element.startsWith("+++")) {
                next = splitFilename(element, visitor, false);
                continue;
            }
            if (element.startsWith("@@")) {
                final Matcher matcher = CHUNK_PATTERN.matcher(element);
                if (matcher.matches()) {
                    final int originalPosition = Integer.parseInt(matcher.group("origpos"));
                    final int revisedPosition = Integer.parseInt(matcher.group("revipos"));
                    String origLen = matcher.group("origlen"), reviLen = matcher.group("revilen");
                    final int originalLength = origLen == null ? 1 : Integer.parseInt(origLen);
                    final int revisedLength = reviLen == null ? 1 : Integer.parseInt(reviLen);
                    next = visitor.visitChunkStart(originalPosition, originalLength, revisedPosition, revisedLength);
                    int original = 0, revised = 0;
                    while ((original < originalLength || revised < revisedLength) && iterator.hasNext()) {
                        element = iterator.next();
                        if (element.length() > 0) {
                            char tag = element.charAt(0);
                            String line = element.substring(1);
                            switch (tag) {
                                case '+':
                                    revised++;
                                    next = visitor.visitLine(DiffRow.Tag.INSERT, line);
                                    break;
                                case '-':
                                    original++;
                                    next = visitor.visitLine(DiffRow.Tag.DELETE, line);
                                    break;
                                default:
                                    original++;
                                    revised++;
                                    next = visitor.visitLine(DiffRow.Tag.EQUAL, line);
                            }
                        } else {
                            original++;
                            revised++;
                            next = visitor.visitLine(DiffRow.Tag.EQUAL, "");
                        }
                        if (!next) break;
                    }
                    if (!next) break;
                    if (original != originalLength || revised != revisedLength)
                        new DiffException("Length mismatch: @@ " + originalPosition + "," + originalLength + " " + revisedPosition + "," + revisedLength + " @@ " + original + "/" + revised).printStackTrace();
                    next = visitor.visitChunkEnd(originalPosition, originalLength, revisedPosition, revisedLength);
                }
            }
        }
    }

    private boolean splitFilename(String element, Visitor visitor, boolean original) throws DiffException {
        Date date = null;
        int idx = element.lastIndexOf("  ");
        if (idx > 0) {
            try {
                date = UnifedDiffDateFormat.INSTANCE.parse(element.substring(idx + 2));
            } catch (ParseException e) {
                e.printStackTrace();
            }
            element = element.substring(3, idx).trim();
        } else {
            element = element.substring(3).trim();
        }
        if (original) {
            return visitor.visitOriginal(element, date);
        } else {
            return visitor.visitRevised(element, date);
        }
    }
}
