/*
   Copyright 2010 Dmitry Naumenko (dm.naumenko@gmail.com)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */
package pw.prok.kdiff;

import pw.prok.kdiff.diff.DiffException;
import pw.prok.kdiff.diff.Equalizer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Holds the information about the part of text involved in the diff process
 * <br>
 * <br>
 * Text is represented as <code>Object[]</code> because the diff engine is
 * capable of handling more than plain ascci. In fact, arrays or lists of any
 * type that implements {@link java.lang.Object#hashCode hashCode()} and
 * {@link java.lang.Object#equals equals()} correctly can be subject to
 * differencing using this library.
 *
 * @param <T> The type of the compared elements in the 'lines'.
 */
public class Chunk<T> {
    private final int position;
    private List<T> lines;

    /**
     * Creates a chunk and saves a copy of affected lines
     *
     * @param position the start position
     * @param lines    the affected lines
     */
    public Chunk(int position, List<T> lines) {
        this.position = position;
        this.lines = lines;
    }

    /**
     * Creates a chunk and saves a copy of affected lines
     *
     * @param position the start position
     * @param lines    the affected lines
     */
    public Chunk(int position, T[] lines) {
        this.position = position;
        this.lines = Arrays.asList(lines);
    }

    /**
     * Verifies that this chunk's saved text matches the corresponding text in
     * the given sequence.
     *
     * @param target the sequence to verify against.
     * @throws PatchFailedException if the patch cannot be applied.
     */
    public void verify(List<T> target) throws PatchFailedException {
        try {
            applyHunk(target);
        } catch (DiffException e) {
            throw new PatchFailedException(e);
        }
    }

    public int applyHunk(List<T> target) throws DiffException {
        return applyHunk(target, 0);
    }

    public int applyHunk(List<T> target, int lastOffset) throws DiffException {
        return applyHunk(target, lastOffset, (Equalizer<T>) Equalizer.DEFAULT_EQUALIZER);
    }

    public int applyHunk(List<T> target, int lastOffset, Equalizer<T> equalizer) throws DiffException {
        final int actual = search(target, lines, position + lastOffset, equalizer);
        if (actual < 0) throw new DiffException("Hunk cannot be applied");
        return actual - position;
    }

    private static <T> int search(List<T> where, List<T> what, int point, Equalizer<T> equalizer) {
        final int start = 0;
        final int end = where.size() - what.size();
        if (end < 0) return -1;
        if (equals(where, what, point, equalizer)) return point;
        for (int i = 1; point - i >= start || point + i <= end; i++) {
            if (point - i >= start && equals(where, what, point - i, equalizer)) return point - i;
            if (point + i <= end && equals(where, what, point + i, equalizer)) return point + i;
        }
        return -1;
    }

    private static <T> boolean equals(List<T> where, List<T> what, int point, Equalizer<T> equalizer) {
        for (int i = 0; i < what.size(); i++) {
            final T o1 = where.get(point + i);
            final T o2 = what.get(i);
            if (!equalizer.equals(o1, o2)) return false;
        }
        return true;
    }

    /**
     * @return the start position of chunk in the text
     */
    public int getPosition() {
        return position;
    }

    public void setLines(List<T> lines) {
        this.lines = lines;
    }

    /**
     * @return the affected lines
     */
    public List<T> getLines() {
        return lines;
    }

    public int size() {
        return lines.size();
    }

    /**
     * @return the index of the last line of the chunk.
     */
    public int last() {
        return getPosition() + size() - 1;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((lines == null) ? 0 : lines.hashCode());
        result = prime * result + position;
        result = prime * result + size();
        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Chunk<T> other = (Chunk) obj;
        if (lines == null) {
            if (other.lines != null)
                return false;
        } else if (!lines.equals(other.lines))
            return false;
        if (position != other.position)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "[position: " + position + ", size: " + size() + ", lines: " + lines + "]";
    }

    /**
     * @return full copy of this chunk
     */
    public Chunk<T> copy() {
        return new Chunk<T>(position, new ArrayList<T>(lines));
    }
}
