package rw.bootstrap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStreamReader;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;

public class RWBootstrap {
    private RWBootstrap() throws IllegalAccessException {
        throw new IllegalAccessException("Cannot create instance of static helper class!");
    }

    private static final Logger LOGGER = LoggerFactory.getLogger(RWBootstrap.class);
    private static final Map<Class<? extends RWApi<?>>, Class<? extends RWConfiguration<?, ?>>> API_MAPPINGS = new HashMap<>();
    private static final String IMPL_FORMAT = "META-INF/rwBootstrap/%s.properties";

    public static <Api extends RWApi<Config>, Config extends RWConfiguration<Config, Api>> Api api(Class<? extends Api> apiClass) {
        return custom(apiClass).build();
    }

    @SuppressWarnings("unchecked")
    public static <Api extends RWApi<Config>, Config extends RWConfiguration<Config, Api>> Config custom(Class<? extends Api> apiClass) {
        if (!RWApi.class.isAssignableFrom(apiClass)) {
            LOGGER.error("{} cannot be casted to {}", apiClass, RWApi.class);
            throw new RuntimeException("Incorrect API class");
        }
        final Class<? extends Config> config = lookupImplementation(apiClass);
        try {
            return config.newInstance();
        } catch (Exception e) {
            LOGGER.error("Error occurred during configuration creation", e);
            throw new RuntimeException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private static <Api extends RWApi<Config>, Config extends RWConfiguration<Config, Api>> Class<? extends Config> lookupImplementation(Class<? extends Api> apiClass) {
        Class<? extends RWConfiguration<?, ?>> implClass = API_MAPPINGS.get(apiClass);
        if (implClass == null) {
            synchronized (API_MAPPINGS) {
                implClass = API_MAPPINGS.get(apiClass);
                if (implClass == null) {
                    implClass = findImplementation(apiClass);
                    if (implClass == null) {
                        throw new RuntimeException("Unable to find implementation for " + apiClass);
                    }
                    API_MAPPINGS.put(apiClass, implClass);
                }
            }
        }
        return (Class<? extends Config>) implClass;
    }

    @SuppressWarnings("unchecked")
    private static <Api extends RWApi<Config>, Config extends RWConfiguration<Config, Api>> Class<? extends RWConfiguration<?, ?>> findImplementation(Class<? extends Api> apiClass) {
        final int modifiers = apiClass.getModifiers();
        final boolean isInterface = Modifier.isInterface(modifiers);
        final boolean isAbstract = Modifier.isAbstract(modifiers);
        LOGGER.debug("{} is {}interface and {}abstract class", apiClass, isInterface ? "" : "not ", isAbstract ? "" : "not ");
        if (!isAbstract && !isInterface) {
            LOGGER.debug("{} recognized as direct implementation");
            return RWBootstrap.<Api, Config>getConfigurationChain(apiClass);
        }
        return recursiveLookup(RWBootstrap.class.getClassLoader(), apiClass);
    }

    private static <Api extends RWApi<Config>, Config extends RWConfiguration<Config, Api>> Class<? extends RWConfiguration<?, ?>> recursiveLookup(ClassLoader classLoader, Class<?> apiClass) {
        if (apiClass == null || !RWApi.class.isAssignableFrom(apiClass)) return null;
        LOGGER.debug("Trying to lookup implementation for {}", apiClass);
        try {
            List<URL> urls = toList(classLoader.getResources(String.format(IMPL_FORMAT, apiClass.getName())));
            if (urls.size() == 0) return null;
            LOGGER.debug("Found following implementations descriptors: {}", urls);
            if (urls.size() > 1)
                LOGGER.warn("{} more than one implementations found, picking first in queue");
            final URL url = urls.get(0);
            final Properties properties = new Properties();
            properties.load(new InputStreamReader(url.openStream(), StandardCharsets.UTF_8));
            final String implClassName = properties.getProperty("implementation-class");
            if (implClassName == null) {
                LOGGER.warn("Missing implementation-class property in descriptor {}", url);
                throw new RuntimeException("Missing implementation-class property in descriptor " + url);
            }
            final Class<?> implClass = Class.forName(implClassName, true, classLoader);
            return RWBootstrap.<Api, Config>getConfigurationChain(implClass);
        } catch (Exception e) {
            LOGGER.error("Error occurred during implementation lookup", e);
        }
        Class<? extends RWConfiguration<?, ?>> config = RWBootstrap.<Api, Config>recursiveLookup(classLoader, apiClass.getSuperclass());
        if (config != null) return config;
        for (Class<?> interfaceClass : apiClass.getInterfaces()) {
            config = RWBootstrap.<Api, Config>recursiveLookup(classLoader, interfaceClass);
            if (config != null) return config;
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private static <Api extends RWApi<Config>, Config extends RWConfiguration<Config, Api>> Class<? extends Config> getConfigurationChain(Class<?> apiClass) {
        final ChainConfiguration config = apiClass.getAnnotation(ChainConfiguration.class);
        if (config == null) {
            LOGGER.error("Found implementation {} with missing ChainConfiguration annotation, fix it!", apiClass);
            throw new RuntimeException("Missing ChainConfiguration annotation for " + apiClass);
        }
        return (Class<? extends Config>) config.value();
    }

    private static <T> List<T> toList(Enumeration<T> enumeration) {
        List<T> list = new ArrayList<>();
        while (enumeration.hasMoreElements())
            list.add(enumeration.nextElement());
        return list;
    }
}
