package pw.ian.sangria_scalapb

import scalapb.descriptors._
import sangria.schema._
import com.trueaccord.scalapb._
import cats._
import cats.implicits._

case class SchemaTransformer(
  renames: Map[String, String] = Map(),
  contextTransforms: Map[
    String,
    List[FieldContext => FieldContext],
  ] = Map(),
  messageFields: Map[String, List[TypeRegistry => Field[Unit, FieldContext]]] = Map(),
  foreignKeys: Map[
    String,
    List[
      (FieldDescriptor, TypeRegistry) => Option[Field[Unit, FieldContext]]
    ],
  ] = Map(),
) {

  def newFieldContext(
    desc: Descriptor,
    pmessage: PMessage,
  ): FieldContext = {
    FieldContext(
      desc = desc,
      pmessage = pmessage,
      transformer = this,
    )
  }

  def newFieldContext[T <: GeneratedMessage with Message[T]](msg: T)(
    implicit comp: GeneratedMessageCompanion[T],
  ): FieldContext = {
    newFieldContext(
      desc = comp.scalaDescriptor,
      pmessage = msg.toPMessage,
    )
  }

  def buildMessageFields(
    registry: TypeRegistry,
    desc: Descriptor,
  ): List[Field[Unit, FieldContext]] = {
    messageFields.get(desc.fullName).orEmpty.map { fn =>
      fn(registry)
    }
  }

  def buildForeignKeys(
    registry: TypeRegistry,
    fd: FieldDescriptor,
  ): List[Field[Unit, FieldContext]] = {
    val fields = foreignKeys.get(fd.name) match {
      case Some(fieldBuilders) => fieldBuilders.map { fb =>
        fb(fd, registry)
      }
      case None => List()
    }
    fields.flatten
  }

}

object SchemaTransformer {
  implicit val schemaTransformer: Monoid[SchemaTransformer] = cats.derive.monoid
}

object RenameTransform {

  def apply[T <: GeneratedMessage with Message[T]](name: String)(
    implicit comp: GeneratedMessageCompanion[T],
    ): SchemaTransformer = {
    SchemaTransformer(renames = Map(comp.scalaDescriptor.fullName -> name))
  }

  def apply[T <: GeneratedEnum](name: String)(
    implicit comp: GeneratedEnumCompanion[T],
    ): SchemaTransformer = {
    SchemaTransformer(renames = Map(comp.scalaDescriptor.fullName -> name))
  }

}

object ContextTransform {

  def apply[T <: GeneratedMessage with Message[T]](transform: FieldContext => FieldContext)(
    implicit comp: GeneratedMessageCompanion[T],
  ): SchemaTransformer = {
    SchemaTransformer(contextTransforms = Map(comp.scalaDescriptor.fullName -> List(transform)))
  }

}

/**
  * Adds a field to a message type.
  */
object AddMessageFieldTransform {

  def apply[T <: GeneratedMessage with Message[T]](
    fieldBuilder: TypeRegistry => Field[Unit, FieldContext],
  )(
    implicit comp: GeneratedMessageCompanion[T],
  ): SchemaTransformer = {
    SchemaTransformer(
      messageFields = Map(comp.scalaDescriptor.fullName -> List(fieldBuilder))
    )
  }

}

/**
  * Adds a "foreign key" field to a message type, colocated with the given field name. (population)
  */
object AddForeignRefFieldTransform {

  def apply(
    foreignKey: String,
    fieldBuilder: (FieldDescriptor, TypeRegistry) => Option[Field[Unit, FieldContext]],
  ): SchemaTransformer = {
    SchemaTransformer(
      foreignKeys = Map(foreignKey -> List(fieldBuilder))
    )
  }


  def applyMany(
    foreignKeys: List[String],
    fieldBuilder: (String, FieldDescriptor, TypeRegistry) => Option[Field[Unit, FieldContext]],
  ): SchemaTransformer = {
    foreignKeys.map(
      fk => apply(
        fk,
        (fd, registry) => fieldBuilder(
          fk, fd, registry))).combineAll
  }

}
