/*
 * HYSTERIA GROUP LICENSE SPECIFICATION (1.2)
 * ================================================
 * An open-source-esque license that seeks to preserve some basic author rights whilst maintaining the integrity of an open-source project
 *
 * This license may also be applied to non-code works such as:
 * - Books
 * - Art
 * - Blueprints
 * - Movies
 *
 *
 * 1.1. Definitions
 *
 *     - THAT OF: Free
 *       - having no expressed cost
 *
 *     - THAT OF: Licensed
 *       - Any work that is expressed to be under this license
 *
 *     - THAT OF: Copyright
 *       - The right of someone to redistribute derivatives (or otherwise) of an original
 *
 *     - THAT OF: Intellectual Property
 *       - 1. The work of an individual, and the sole work thereof
 *       - 2. The work of a group of individuals, and the sole work thereof
 *
 * 1.2. Code Disclaimer(s)
 *
 *     1 ANY LICENSED code is provided as is
 *       .1 It is due to this that SHOULD any code within the LICENSED projects be in ANY way responsible for damages to a machine, or person(s) the AUTHOR is in no way responsible
 *       .2 This SHOULD NOT be NEGATED by an AUTHOR who has PUBLISHED under this LICENSE
 *
 * 1.3. License Restrictions
 *
 *     1. IN REGARDS TO: Licensing of derivatives and/or uses of the LICENSED work(s)
 *        .1 ANY derivative works MUST share the HGLSv*
 *        .2 ANY works that MAKE USE of the LICENSED work(s) must be published under the HGLSv*, a similar license, or a beerware-style license (so long as it preserves the HGLSv* values)
 *
 *     2. IN REGARDS TO: Cost
 *       .1 ALL derivative works MUST be free - no fees may be charged, unless you receive EXPRESSED PERMISSION from the AUTHOR
 *       .2 ANY works that MAKE USE of the LICENSED work(s) must also be FREE - no fees may be charged.
 *
 *     3. IN REGARDS TO: Intellectual property
 *        .N THIS license seeks to preserve the intellectual property of AUTHORS, whilst allowing such property to be freely re-used and distributed.
 *           IN ORDER to accomplish this, all USES AND DERIVATIVES must meet the following conditions
 *
 *       1. CREDIT FOR the work of ANY one AUTHOR must be preserved.
 *         .1 IF NO AUTHOR is specified DO NOT TAKE CREDIT. THIS is PUBLIC DOMAIN, and should REMAIN THAT WAY.
 *         .2 IF an AUTHOR is SPECIFIED, DO NOT CHANGE THE SPECIFIED AUTHOR's name UNDER ANY CONDITIONS
 *
 *       2. IF ADDING WORK of OTHER AUTHORS
 *          .1 INSURE that you are PERMITTED to DO SO
 *
 *       3. WHEN ADDING YOUR own WORK to a GROUP WORK
 *         .1 INSURE that YOU TAKE CREDIT for it
 *           .1 OTHERWISE, you are subject to section 1.3. , sub-section 3. , clause .1
 *         .2 SHOULD you be contributing to a LICENSED group work THAT
 *           .1 THE WORK shall remain under the HGLSv* LICENSE applied TO the GROUP WORK
 *           .2 IT IS THE DISCRETION of the PROJECT MANAGER whether or not YOUR WORK may be RELEASED from the LICENSE upon YOUR SOLE REQUEST
 *           .3 THE WORK shall BE IN YOUR NAME
 *              .1 SHOULD THE WORK be attributed to another FOR ANY REASON - that it is STILL YOUR WORK
 *                 .1 HOWEVER - YOU MUST speak with the PROJECT MANAGER in regards to the aforementioned MODIFICATION
 *
 *       4. AS AN AUTHOR (Author Rights)
 *          .1 YOU MAY
 *             .1 PUBLISH YOUR WORK under ANOTHER LICENSE, PURSUANT TO the RESTRICTIONS SPECIFIED under section 1.3., sub-section 1., clause .1
 *             .2 INCLUDE identical work IN OTHER GROUP PROJECTS
 *             .3 USE identical work IN YOUR OWN PROJECTS
 *             .4 GRANT PERMISSION for COMMERCIAL USE under YOUR CONDITIONS
 *                .1 SO LONG AS:
 *                   .1 The GRANTEE does not enact restrictions on FREE USE of YOUR WORK in any FASHION
 *                   .2 The GRANTEE ATTRIBUTES to you YOUR WORK
 *          .2 YOU MAY NOT
 *             .1 PUBLISH for PROFIT the works of OTHER AUTHORS as they are INCLUDED in GROUP PROJECTS that YOU MANAGE
 *             .2 CLAIM as YOURS the WORKS of OTHER AUTHORS as they are INCLUDED in GROUP PROJECTS that YOU MANAGE
 */

package pw.hysteria.input.dashfoo.command;

import pw.hysteria.input.dashfoo.command.parametertransforms.*;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Creator: KingBowser
 * Date: 5/26/13
 * Time: 7:39 PM
 * Refer to LICENSE for licensing information
 */
public final class MethodAsInvokableFactory {

    /**
     * Defines the appropriate parameter transforms to assign to each method parameter
     */
    private final Map<Class<?>, ParamTypeTransform<?>> transformMap = new HashMap<>();

    public MethodAsInvokableFactory() {

        /*
            Register default transforms (array and single)
         */

        registerTransform(String.class, new StringTransform());
        registerTransform(Boolean.class, new BooleanTransform());

        registerTransform(Byte.class, new ByteTransform());
        registerTransform(Short.class, new ShortTransform());
        registerTransform(Integer.class, new IntegerTransform());
        registerTransform(Long.class, new LongTransform());
        registerTransform(Float.class, new FloatTransform());
        registerTransform(Double.class, new DoubleTransform());

        registerTransform(boolean.class, new BooleanTransform());

        registerTransform(byte.class, new ByteTransform());
        registerTransform(short.class, new ShortTransform());
        registerTransform(int.class, new IntegerTransform());
        registerTransform(long.class, new LongTransform());
        registerTransform(float.class, new FloatTransform());
        registerTransform(double.class, new DoubleTransform());

        /*
            The generics API prevents us from transforming primitive arrays without mucking up our code.
         */

        registerTransform(String[].class, new ArrayTransformWrapper<>(new StringTransform()));
        registerTransform(Boolean[].class, new ArrayTransformWrapper<>(new BooleanTransform()));

        registerTransform(Byte[].class, new ArrayTransformWrapper<>(new ByteTransform()));
        registerTransform(Short[].class, new ArrayTransformWrapper<>(new ShortTransform()));
        registerTransform(Integer[].class, new ArrayTransformWrapper<>(new IntegerTransform()));
        registerTransform(Long[].class, new ArrayTransformWrapper<>(new LongTransform()));
        registerTransform(Float[].class, new ArrayTransformWrapper<>(new FloatTransform()));
        registerTransform(Double[].class, new ArrayTransformWrapper<>(new DoubleTransform()));

    }

    /**
     * Register a transform for a class
     *
     * @param ptClass   class of parameter
     * @param ptFactory factory to produce instance of parameter type
     * @param <PT>      type of parameter
     */
    public final <PT> void registerTransform(Class<PT> ptClass, ParamTypeTransform<PT> ptFactory) {

        if (transformMap.containsKey(ptClass))
            throw new IllegalArgumentException(ptClass.getName() + " already has a transform registered. Please unregister it first.");

        transformMap.put(ptClass, ptFactory);

    }

    /**
     * Remove a mapping for a ParameterTypeTransform
     *
     * @param ptClass class of the parameter type
     */
    @SuppressWarnings("unused")
    public void unregisterTransform(Class<?> ptClass) {

        if (!transformMap.containsKey(ptClass))
            throw new IllegalArgumentException(ptClass.getName() + " is not registered to a transform. Please register it first.");

        transformMap.remove(ptClass);

    }

    @SuppressWarnings("unchecked")
    public <PT> ParamTypeTransform<PT> translateParamToTransform(Class<PT> paramClass) {

        if (!transformMap.containsKey(paramClass))
            throw new IllegalArgumentException("No such parameter mapping exists");

        return (ParamTypeTransform<PT>) transformMap.get(paramClass);
    }

    /**
     * Construct a MethodAsFlag from any given method
     *
     * @param handles     handles
     * @param method      method to be invoked
     * @param methodOwner instance of the class that owns the method
     * @return compiled MethodAsFlag
     */
    private MethodAsFlag constructFromMethod(String[] handles, Method method, Object methodOwner) {

        if (methodOwner == null)
            throw new IllegalArgumentException("Method owner is null");

        final ArrayList<ParamTypeTransform<?>> transforms = new ArrayList<>();

        for (Class<?> paramType : method.getParameterTypes()) {
            try {
                transforms.add(translateParamToTransform(paramType));
            } catch (IllegalArgumentException exception) {
                throw new IllegalStateException("Provided method contains unrecognized parameter types");
            }
        }

        return new MethodAsFlag(handles, transforms, method, methodOwner);

    }

    /**
     * Construct a MethodAsFlag from an @IsFlag annotated method
     *
     * @param method      annotated method
     * @param methodOwner container class instance
     * @return compiled MethodAsFlag
     */
    public MethodAsFlag constructFromMethod(Method method, Object methodOwner) {

        if (!method.isAnnotationPresent(IsFlag.class))
            throw new IllegalArgumentException("Provided method is not annotated with @IsFlag");

        return constructFromMethod(method.getAnnotation(IsFlag.class).handles(), method, methodOwner);

    }

    /**
     * Construct an ArrayList of MethodsAsInvokable from an instance of a class
     *
     * @param methodsOwner class instance
     * @return enumerated methods
     */
    public ArrayList<MethodAsFlag> constructFromClass(Object methodsOwner) {

        final ArrayList<MethodAsFlag> compiledMethods = new ArrayList<>();

        /*
            Iterate through methods and spot the methods with @IsFlag
         */
        for (Method method : methodsOwner.getClass().getMethods()) {
            if (method.isAnnotationPresent(IsFlag.class)) {
                compiledMethods.add(constructFromMethod(method, methodsOwner));
            }
        }

        return compiledMethods;
    }


    /**
     * Defines an invokable that has been constructed based on a method signature
     *
     * WARNING: MESSY CODE ZONE
     *
     * Creator: KingBowser
     * Date: 5/26/13
     * Time: 7:32 PM
     * Refer to LICENSE for licensing information
     */
    public static final class MethodAsFlag implements Flag, HelpProvider {

        /**
         * Defines the argument types used at each position
         */
        private final List<ParamTypeTransform<?>> paramTypeTransforms;

        /**
         * The method to invoke
         */
        private final Method invokeMethod;

        /**
         * The instance of the method owner to invoke the method against
         */
        private final Object invokeAgainst;

        /**
         * The handle of the invokable
         */
        private final String[] handles;

        /**
         * Default Constructor
         *
         * @param acceptableFlags         array of possible handles
         * @param typeTransforms          list of transforms in the order that they will be needed
         * @param methodToInvoke          method to invokeFlag
         * @param methodContainerInstance an instance of the object containing the invoking method
         */
        MethodAsFlag(String[] acceptableFlags, List<ParamTypeTransform<?>> typeTransforms, Method methodToInvoke,
                     Object methodContainerInstance) {

            this.handles = acceptableFlags;

            this.paramTypeTransforms = typeTransforms;
            this.invokeMethod = methodToInvoke;
            this.invokeAgainst = methodContainerInstance;


        }

        @Override
        public String[] getHandles() {
            return handles;
        }

        @Override
        public void invokeFlag(String... args) throws FlagInvocationException {

            /*
                Are there too many args?
            */
            if (args.length > paramTypeTransforms.size())
                throw new FlagInvocationException("Number of arguments passed exceeds number of expected arguments", this, args);

            /*
                Transform our arguments into the appropriate format
            */
            Object[] methodArgs = new Object[args.length];


            int positionInParams = 0;

            for (String arg : args) {
                methodArgs[positionInParams] = paramTypeTransforms.get(positionInParams).transform(arg);
                positionInParams++;
            }

            /*
                Make sure we can invoke from this scope (JIC)
            */
            invokeMethod.setAccessible(true);

            /*
                Method return trap
            */
            Object methodReturn;

            try {
                methodReturn = invokeMethod.invoke(invokeAgainst, methodArgs);
            } catch (IllegalAccessException e) {
                throw new IllegalStateException("Unable to access invocation target from current scope");
            } catch (InvocationTargetException e) {
                throw new IllegalArgumentException("Passed an invalid method container during construction");
            }

        }

        @Override
        public int getArity() {
            return paramTypeTransforms.size();
        }

        @Override
        public final HelpProvider getHelpProvider() {
            return this;
        }

        @Override
        public final String getDescription() {
            return null;
        }

        @Override
        public final String getHelp() {
            return null;
        }
    }

}
