package pw.aru.utils;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Clean version of {@link java.util.Properties}, using new stuff like HashMaps and Generics.
 * Compatible with files generated by the the original Properties class.
 *
 * @author Arthur van Hoff
 * @author Michael McCloskey
 * @author Xueming Shen
 * @author AdrianTodt
 * @see java.util.Properties
 */
public class Properties extends ConcurrentHashMap<String, String> {
    private class LineReader {
        byte[] inByteBuf;
        char[] inCharBuf;
        int inLimit = 0;
        int inOff = 0;
        InputStream inStream;
        char[] lineBuf = new char[1024];
        Reader reader;

        LineReader(InputStream inStream) {
            this.inStream = inStream;
            inByteBuf = new byte[8192];
        }

        LineReader(Reader reader) {
            this.reader = reader;
            inCharBuf = new char[8192];
        }

        int readLine() throws IOException {
            int len = 0;
            char c;

            boolean skipWhiteSpace = true;
            boolean isCommentLine = false;
            boolean isNewLine = true;
            boolean appendedLineBegin = false;
            boolean precedingBackslash = false;
            boolean skipLF = false;

            while (true) {
                if (inOff >= inLimit) {
                    inLimit = (inStream == null) ? reader.read(inCharBuf) : inStream.read(inByteBuf);
                    inOff = 0;
                    if (inLimit <= 0) {
                        if (len == 0 || isCommentLine) return -1;
                        if (precedingBackslash) len--;
                        return len;
                    }
                }
                c = inStream != null ? (char) (0xff & inByteBuf[inOff++]) : inCharBuf[inOff++];
                if (skipLF) {
                    skipLF = false;
                    if (c == '\n') continue;
                }
                if (skipWhiteSpace) {
                    if (c == ' ' || c == '\t' || c == '\f' || !appendedLineBegin && (c == '\r' || c == '\n')) continue;
                    skipWhiteSpace = false;
                    appendedLineBegin = false;
                }
                if (isNewLine) {
                    isNewLine = false;
                    if (c == '#' || c == '!') {
                        isCommentLine = true;
                        continue;
                    }
                }

                if (c != '\n' && c != '\r') {
                    lineBuf[len++] = c;
                    if (len == lineBuf.length) {
                        int newLength = lineBuf.length * 2;
                        if (newLength < 0) newLength = Integer.MAX_VALUE;
                        char[] buf = new char[newLength];
                        System.arraycopy(lineBuf, 0, buf, 0, lineBuf.length);
                        lineBuf = buf;
                    }
                    //flip the preceding backslash flag
                    precedingBackslash = c == '\\' && !precedingBackslash;
                } else {
                    // reached EOL
                    if (isCommentLine || len == 0) {
                        isCommentLine = false;
                        isNewLine = true;
                        skipWhiteSpace = true;
                        len = 0;
                        continue;
                    }
                    if (inOff >= inLimit) {
                        inLimit = (inStream == null) ? reader.read(inCharBuf) : inStream.read(inByteBuf);
                        inOff = 0;
                        if (inLimit <= 0) {
                            if (precedingBackslash) len--;
                            return len;
                        }
                    }
                    if (precedingBackslash) {
                        len -= 1;
                        //skip the leading whitespace characters in following line
                        skipWhiteSpace = true;
                        appendedLineBegin = true;
                        precedingBackslash = false;
                        if (c == '\r') skipLF = true;
                    } else {
                        return len;
                    }
                }
            }
        }
    }

    private static final char[] hexDigit = {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
    };

    /**
     * Loads the {@link Properties} from the file system.
     *
     * @param file File to read properties from.
     * @return A {@link Properties} instance loaded with the content from the file.
     * @throws IOException if an IOException occurs.
     */
    @NotNull
    public static Properties fromFile(@NotNull File file) throws IOException {
        Properties p = new Properties();
        p.load(new FileInputStream(file));
        return p;
    }

    /**
     * Loads the {@link Properties} from the file system.
     *
     * @param file File to read properties from.
     * @return A new {@link Properties} instance loaded with the content from the file.
     * @throws IOException if an IOException occurs.
     */
    @NotNull
    public static Properties fromFile(@NotNull String file) throws IOException {
        return fromFile(new File(file));
    }

    /**
     * Loads the {@link Properties} from a string.
     *
     * @param string String to read properties from.
     * @return A {@link Properties} instance loaded with the content from the string.
     * @throws IOException if an IOException occurs.
     */
    @NotNull
    public static Properties fromString(@NotNull String string) throws IOException {
        Properties p = new Properties();
        p.loadFromString(string);
        return p;
    }

    /**
     * Loads values from the Reader.
     *
     * @param reader Reader to read the values from.
     * @throws IOException if an IOException occurs.
     */
    public synchronized void load(@NotNull Reader reader) throws IOException {
        _load(new LineReader(reader));
    }

    /**
     * Loads values from the InputStream.
     *
     * @param inStream InputStream to read the values from.
     * @throws IOException if an IOException occurs.
     */
    public synchronized void load(@NotNull InputStream inStream) throws IOException {
        _load(new LineReader(inStream));
    }

    /**
     * Loads values from a String.
     *
     * @param string Reader to read the values from.
     * @throws IOException if an IOException occurs.
     */
    public synchronized void loadFromString(@NotNull String string) throws IOException {
        _load(new LineReader(new StringReader(string)));
    }

    /**
     * Stores values into a OutputStream.
     *
     * @param out      OutputStream to write the values to.
     * @param comments (Optional) Comments written on the beginning of the OutputStream.
     * @throws IOException if an IOException occurs.
     */
    public synchronized void store(@NotNull OutputStream out, @Nullable String comments) throws IOException {
        _store(
            new BufferedWriter(new OutputStreamWriter(out, "8859_1")),
            comments,
            true
        );
    }

    /**
     * Stores values into a Writer.
     *
     * @param writer   Writer to write the values to.
     * @param comments (Optional) Comments written on the beginning of the Writer.
     * @throws IOException if an IOException occurs.
     */
    public synchronized void store(@NotNull Writer writer, @Nullable String comments) throws IOException {
        _store(
            (writer instanceof BufferedWriter) ? (BufferedWriter) writer : new BufferedWriter(writer),
            comments,
            false
        );
    }

    /**
     * Stores values into a File.
     *
     * @param file     File to write the values to.
     * @param comments (Optional) Comments written on the beginning of the File.
     * @throws IOException if an IOException occurs.
     */
    public synchronized void store(@NotNull File file, @Nullable String comments) throws IOException {
        _store(
            new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), "8859_1")),
            comments,
            true
        );
    }

    /**
     * Stores values into a String.
     *
     * @param comments (Optional) Comments written on the beginning of the File.
     * @return The values written to a String.
     * @throws IOException if an IOException occurs.
     */
    @NotNull
    public synchronized String storeToString(@Nullable String comments) throws IOException {
        StringWriter w = new StringWriter();
        _store(new BufferedWriter(w), comments, false);
        return w.toString();
    }

    private static char toHex(int nibble) {
        return hexDigit[(nibble & 0xF)];
    }

    private static void writeComments(BufferedWriter bw, String comments) throws IOException {
        bw.write("#");
        char[] in = comments.toCharArray();
        int len = comments.length();
        int current = 0;
        int last = 0;
        char[] uu = new char[6];
        uu[0] = '\\';
        uu[1] = 'u';
        while (current < len) {
            char c = in[current];
            if (c > '\u00ff' || c == '\n' || c == '\r') {
                if (last != current)
                    bw.write(comments.substring(last, current));
                if (c > '\u00ff') {
                    uu[2] = toHex((c >> 12) & 0xf);
                    uu[3] = toHex((c >> 8) & 0xf);
                    uu[4] = toHex((c >> 4) & 0xf);
                    uu[5] = toHex(c & 0xf);
                    bw.write(new String(uu));
                } else {
                    bw.newLine();
                    if (c == '\r' &&
                        current != len - 1 &&
                        in[current + 1] == '\n') {
                        current++;
                    }
                    if (current == len - 1 || (in[current + 1] != '#' && in[current + 1] != '!'))
                        bw.write("#");
                }
                last = current + 1;
            }
            current++;
        }
        if (last != current) bw.write(comments.substring(last, current));
        bw.newLine();
    }

    private void _load(LineReader lr) throws IOException {
        char[] convtBuf = new char[1024];
        int limit;
        int keyLen;
        int valueStart;
        char c;
        boolean hasSep;
        boolean precedingBackslash;

        while ((limit = lr.readLine()) >= 0) {
            keyLen = 0;
            valueStart = limit;
            hasSep = false;

            precedingBackslash = false;
            while (keyLen < limit) {
                c = lr.lineBuf[keyLen];
                //need check if escaped.
                if ((c == '=' || c == ':') && !precedingBackslash) {
                    valueStart = keyLen + 1;
                    hasSep = true;
                    break;
                } else if ((c == ' ' || c == '\t' || c == '\f') && !precedingBackslash) {
                    valueStart = keyLen + 1;
                    break;
                }
                precedingBackslash = c == '\\' && !precedingBackslash;
                keyLen++;
            }
            while (valueStart < limit) {
                c = lr.lineBuf[valueStart];
                if (c != ' ' && c != '\t' && c != '\f') {
                    if (!hasSep && (c == '=' || c == ':')) {
                        hasSep = true;
                    } else {
                        break;
                    }
                }
                valueStart++;
            }
            put(loadConvert(lr.lineBuf, 0, keyLen, convtBuf), loadConvert(lr.lineBuf, valueStart, limit - valueStart, convtBuf));
        }
    }

    private void _store(BufferedWriter bw, String comments, boolean escUnicode) throws IOException {
        if (comments != null) writeComments(bw, comments);
        bw.write("#" + new Date().toString());
        bw.newLine();
        synchronized (this) {
            for (Entry<String, String> entry : entrySet()) {
                bw.write(saveConvert(entry.getKey(), true, escUnicode) + "=" + saveConvert(entry.getValue(), false, escUnicode));
                bw.newLine();
            }
        }
        bw.flush();
    }

    private String loadConvert(char[] in, int off, int len, char[] convtBuf) {
        char[] out = convtBuf.length < len ? new char[len * 2] : convtBuf;
        char c;
        int outLen = 0;
        int end = off + len;

        while (off < end) {
            c = in[off++];
            if (c == '\\') {
                c = in[off++];
                if (c == 'u') {
                    // Read the xxxx
                    int value = 0;
                    for (int i = 0; i < 4; i++) {
                        c = in[off++];
                        switch (c) {
                            case '0':
                            case '1':
                            case '2':
                            case '3':
                            case '4':
                            case '5':
                            case '6':
                            case '7':
                            case '8':
                            case '9':
                                value = (value << 4) + c - '0';
                                break;
                            case 'a':
                            case 'b':
                            case 'c':
                            case 'd':
                            case 'e':
                            case 'f':
                                value = (value << 4) + 10 + c - 'a';
                                break;
                            case 'A':
                            case 'B':
                            case 'C':
                            case 'D':
                            case 'E':
                            case 'F':
                                value = (value << 4) + 10 + c - 'A';
                                break;
                            default:
                                throw new IllegalArgumentException("Malformed \\uxxxx encoding.");
                        }
                    }
                    out[outLen++] = (char) value;
                } else {
                    c = c == 't' ? '\t' : c == 'r' ? '\r' : c == 'n' ? '\n' : c == 'f' ? '\f' : c;
                    out[outLen++] = c;
                }
            } else {
                out[outLen++] = c;
            }
        }
        return new String(out, 0, outLen);
    }

    private String saveConvert(String s, boolean escapeSpace, boolean escapeUnicode) {
        char[] in = s.toCharArray();
        int len = in.length;
        StringBuilder b = new StringBuilder(len * 2);

        for (int i = 0; i < len; i++) {
            char c = in[i];
            // Handle common case first, selecting largest block that
            // avoids the specials below
            if ((c > 61) && (c < 127)) {
                if (c == '\\') {
                    b.append('\\').append('\\');
                    continue;
                }
                b.append(c);
                continue;
            }
            switch (c) {
                case ' ':
                    if (i == 0 || escapeSpace) b.append('\\');
                    b.append(' ');
                    break;
                case '\t':
                    b.append('\\').append('t');
                    break;
                case '\n':
                    b.append('\\').append('n');
                    break;
                case '\r':
                    b.append('\\').append('r');
                    break;
                case '\f':
                    b.append('\\').append('f');
                    break;
                case '=': // Fall through
                case ':': // Fall through
                case '#': // Fall through
                case '!':
                    b.append('\\').append(c);
                    break;
                default:
                    if (((c < 0x0020) || (c > 0x007e)) & escapeUnicode) {
                        b.append('\\').append('u')
                            .append(toHex((c >> 12) & 0xF))
                            .append(toHex((c >> 8) & 0xF))
                            .append(toHex((c >> 4) & 0xF))
                            .append(toHex(c & 0xF));
                    } else {
                        b.append(c);
                    }
            }
        }
        return b.toString();
    }

}
