/*
 * Copyright (c) 2020-2024 IBA Group.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v2.0 which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-v20.html
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   IBA Group
 *   Zowe Community
 */

package org.zowe.kotlinsdk.zowe.client.sdk.zostso

import org.zowe.kotlinsdk.*
import org.zowe.kotlinsdk.zowe.client.sdk.core.ZOSConnection
import org.zowe.kotlinsdk.zowe.client.sdk.zostso.input.SendTsoParams
import okhttp3.Credentials
import okhttp3.OkHttpClient
import retrofit2.Response
import kotlin.jvm.Throws

/**
 * Class to handle sending data to TSO
 */
class SendTso(
  var connection: ZOSConnection,
  var httpClient: OkHttpClient = UnsafeOkHttpClient.unsafeOkHttpClient
) {

  init {
    connection.checkConnection()
  }

  var response: Response<*>? = null

  /**
   * Create Response
   *
   * @param responses responses from CollectedResponses object
   * @return [SendResponse]
   */
  @Throws(Exception::class)
  private fun createResponse(responses: CollectedResponses): SendResponse {
    return SendResponse(
      true,
      responses.tsos,
      responses.messages ?: throw Exception("no responses messages exist")
    )
  }

  /**
   * Collects responses from address space until it reaches prompt
   *
   * @param tsoResponse object from first Tso response from witch responses are needed, see [TsoResponse]
   * @return [CollectedResponses] response object
   * @throws Exception error executing command
   */
  @Throws(Exception::class)
  fun getAllResponses(tsoResponse: TsoResponse, failOnPrompt: Boolean): CollectedResponses {
    var tso = tsoResponse
    var done = false
    val messages = StringBuilder()
    val tsos = mutableListOf<TsoResponse>()
    tsos.add(tso)
    while(!done) {
      if (tso.tsoData.isNotEmpty()) {
        tso.tsoData.forEach {
          if (it.tsoMessage != null) {
            val tsoMsg = it.tsoMessage
            tsoMsg?.data?.let { data ->
              messages.append(data)
              messages.append("\n")
            }
          } else if (it.tsoPrompt != null) {
            if (messages.toString().contains("IKJ56602I COMMAND SYSTEM RESTARTING DUE TO ERROR")) {
              val IKJ56602I = "IKJ56602I COMMAND SYSTEM RESTARTING DUE TO ERROR"
              val msg = messages.toString()
              val startIndex = msg.indexOf("IKJ56602I")
              messages.delete(startIndex, startIndex + IKJ56602I.length + "\nREADY".length)
              return@forEach
            } else if (messages.isNotEmpty() && messages.toString().contains("READY")) {
              done = true
              return@forEach
            }
            if (failOnPrompt) {
              throw Exception(messages.toString() + "Failed to interact with TSO console: The console requested input, but the 'failOnPrompt'" +
                      " option is set to true, indicating that interactive prompts should be avoided.")
            }
            // TSO PROMPT reached without getting any data, retrying
          }
        }
      }
      if (!done) {
        tso.servletKey?.let { tso = getDataFromTso(it) } ?: throw Exception("servlet key missing")
        tsos.add(tso)
      }
    }
    return CollectedResponses(tsos, messages.toString())
  }

  /**
   * Retrieve tso http request response
   *
   * @param servletKey key of tso address space
   * @return z/OSMF [TsoResponse]
   * @throws Exception error executing command
   */
  @Throws(Exception::class)
  private fun getDataFromTso(servletKey: String): TsoResponse {
    val url = "${connection.protocol}://${connection.host}:${connection.zosmfPort}"
    val tsoApi = buildApi<TsoApi>(url, httpClient)
    val call = tsoApi.receiveMessagesFromTso(
      authorizationToken = Credentials.basic(connection.user, connection.password),
      servletKey = servletKey
    )
    response = call.execute()
    validateResponse(response, "Follow up TSO Messages from TSO command cannot be retrieved")
    return response?.body() as TsoResponse? ?: throw Exception("No body returned")
  }

  /**
   * API method to send data to already started TSO address space, but will read TSO data until a PROMPT is reached.
   *
   * @param command    to send to the TSO address space.
   * @param servletKey returned from a successful start
   * @return [SendResponse] object
   * @throws Exception error executing command
   */
  @Throws(Exception::class)
  fun sendDataToTSOCollect(servletKey: String, command: String, failOnPrompt: Boolean): SendResponse {
    if (servletKey.isEmpty()) {
      throw Exception("servletKey not specified")
    }
    if (command.isEmpty()) {
      throw Exception("command  not specified")
    }

    val putResponse = sendDataToTSOCommon(
      SendTsoParams(servletKey = servletKey, data = command)
    )

    val responses = getAllResponses(putResponse, failOnPrompt)
    return createResponse(responses)
  }

  /**
   * API method to send data to already started TSO address space
   *
   * @param commandParams [SendTsoParams] object with required parameters
   * @return [TsoResponse] object
   * @throws Exception error executing command
   */
  @Throws(Exception::class)
  fun sendDataToTSOCommon(commandParams: SendTsoParams): TsoResponse {
    if (commandParams.data.isEmpty()) {
      throw Exception("commandParams data not specified")
    }
    if (commandParams.servletKey.isEmpty()) {
      throw Exception("commandParams servletKey not specified")
    }

    val url = "${connection.protocol}://${connection.host}:${connection.zosmfPort}"
    val tsoApi = buildApi<TsoApi>(url, httpClient)
    val call = tsoApi.sendMessageToTso(
      authorizationToken = Credentials.basic(connection.user, connection.password),
      body = TsoData(
        tsoResponse = MessageType(version = "0100", data = commandParams.data)
      ),
      servletKey = commandParams.servletKey
    )
    response = call.execute()
    validateResponse(response, "No results from executing tso command after getting TSO address space")
    return response?.body() as TsoResponse? ?: throw Exception("No body returned")
  }

}
