/*
 * Copyright 2015 Allette Systems (Australia)
 * http://www.allette.com.au
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * ---------- Original copyright notice for this portion of the code ----------
 *
 * Adapted from work by Christophe Lauret and Willy Ekasalim
 *
 * Open Source Initiative OSI - The MIT License:Licensing
 * [OSI Approved License]
 *
 * The MIT License
 *
 * Copyright (c) 2008 Rick Jelliffe, Topologi Pty. Ltd, Allette Systems
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.pageseeder.schematron;

import java.io.*;

import org.pageseeder.schematron.svrl.*;

/**
 * Stores the results of the schematron validation.
 *
 * @author Christophe lauret
 * @author Willy Ekasalim
 *
 * @version 2.0
 * @since 1.0
 */
public final class SchematronResult implements Serializable {

  /**
   * The source file name or systemID.
   */
  private final String systemID;

  /**
   * The SVRL content as a string.
   */
  private SVRLData svrl;

  /**
   * Number of failed assert found in results.
   */
  private int assertsCount = -1;

  /**
   * Number of successful reports found in results.
   */
  private int reportsCount = -1;

  /**
   * Constructor of SchematronResult that accept the source file name (or systemID)
   *
   * @param systemID The system ID of the XML for which this result instance is built.
   */
  public SchematronResult(String systemID) {
    this.systemID = systemID;
  }

  /**
   * Constructor of SchematronResult that accept the source file name (or systemID)
   *
   * @param systemID The system ID of the XML for which this result instance is built.
   */
  public SchematronResult(String systemID, SVRLData svrl, int assertsCount, int reportsCount) {
    this.systemID = systemID;
    this.svrl = svrl;
    this.assertsCount = assertsCount;
    this.reportsCount = reportsCount;
  }

  /**
   * @return this object's systemID.
   */
  public String getSystemID() {
    return this.systemID;
  }

  /**
   * @return <code>true</code> if there's no failed assertion;
   *         <code>false</code> if there is failed assertion
   */
  public boolean isValid() {
    return this.assertsCount == 0;
  }

  /**
   * @return <code>true</code> if there's no failed assertion;
   *         <code>false</code> if there is failed assertion
   */
  public boolean hasAsserts() {
    return this.assertsCount > 0;
  }

  /**
   * @return <code>true</code> if there's no failed assertion;
   *         <code>false</code> if there is failed assertion
   */
  public boolean hasReports() {
    return this.reportsCount > 0;
  }

  /**
   * Setter for SVRL content/file.
   *
   * @param svrl The corresponding generated by SVRL.
   */
  @Deprecated
  public void setSVRL(String svrl, int assertsCount, int reportsCount) {
    this.svrl = new SVRLString(svrl);
    this.assertsCount = assertsCount;
    this.reportsCount = reportsCount;
  }

  /**
   * Parse the SVRL output and generate the corresponding SchematronOutput instance.
   *
   * @return the corresponding SchematronOutput instance.
   * @throws SchematronException If any error occurs during parsing.
   */
  public SchematronOutput toSchematronOutput() throws SchematronException {
    try {
      return SVRLParser.parse(this.svrl.getReader());
    } catch (IOException ex) {
      throw new SchematronException(ex);
    }
  }

  /**
   * @return SVRL content as String representation.
   */
  public String getSVRLAsString() {
    return this.svrl.asString();
  }

  /**
   * @return SVRL content as a byte array.
   */
  public byte[] getSVRLAsBytes() {
    return this.svrl.asByteArray();
  }

  /**
   * @return Reader on the SVRL content.
   */
  public Reader getSVRLReader() throws IOException {
    return this.svrl.getReader();
  }

  /**
   * Return the failed assertion message only.
   *
   * <p>This method is used for Pageseeder Schematron Validation error output.
   *
   * @return a concatenation of all failed messages.
   */
  public String getFailedMessage() {
    StringBuilder out = new StringBuilder();
    if (this.assertsCount > 0) {
      SchematronOutput output = toSchematronOutputSilently();
      for (AssertOrReport failedAssertion : output.getFailedAsserts()) {
        out.append(failedAssertion.toMessageString());
      }
    }
    return out.toString();
  }

  // private helpers
  // ----------------------------------------------------------------------------------------------

  /**
   * Parse the SVRL content to extract any failed or success message.
   *
   * <p>The message will be stored in failedAssertions and successfulReports by SVRL handler.
   */
  private SchematronOutput toSchematronOutputSilently() {
    try {
      return SVRLParser.parse(this.svrl.getReader());
    } catch (IOException ex) {
      throw new UncheckedIOException(ex);
    } catch (SchematronException ex) {
      throw new IllegalStateException("Invalid SVRL content", ex);
    }
  }

  static class Builder {

    /**
     * The source file name or systemID.
     */
    private String systemID;

    /**
     * The SVRL content as a string.
     */
    private SVRLData svrl;

    /**
     * Number of failed assert found in results.
     */
    private int assertsCount = -1;

    /**
     * Number of successful reports found in results.
     */
    private int reportsCount = -1;

    public Builder setSystemID(String systemID) {
      this.systemID = systemID;
      return this;
    }

    public Builder setSVRL(SVRLData svrl) {
      this.svrl = svrl;
      return this;
    }

    public Builder setAssertsCount(int assertsCount) {
      this.assertsCount = assertsCount;
      return this;
    }

    public Builder setReportsCount(int reportsCount) {
      this.reportsCount = reportsCount;
      return this;
    }

    public SchematronResult build() {
      return new SchematronResult(this.systemID, svrl, assertsCount, reportsCount);
    }
  }
}
