/*
 * Copyright 2015 Allette Systems (Australia)
 * http://www.allette.com.au
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * ---------- Original copyright notice for this portion of the code ----------
 *
 * Adapted from work by Christophe Lauret and Willy Ekasalim
 *
 * Open Source Initiative OSI - The MIT License:Licensing
 * [OSI Approved License]
 *
 * The MIT License
 *
 * Copyright (c) 2008 Rick Jelliffe, Topologi Pty. Ltd, Allette Systems
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.pageseeder.schematron;

import java.io.PrintStream;
import java.io.StringReader;
import java.util.List;

import org.pageseeder.schematron.svrl.AssertOrReport;
import org.pageseeder.schematron.svrl.SVRL;
import org.pageseeder.schematron.svrl.SVRLParser;
import org.pageseeder.schematron.svrl.SchematronOutput;

/**
 * Stores the results of the schematron validation.
 *
 * @author Christophe lauret
 * @author Willy Ekasalim
 *
 * @version 2.0
 * @since 1.0
 */
public final class SchematronResult {

  /** The source file name or systemID */
  private final String systemID;

  /** The SVRL content as a string */
  private String svrl;

  /**
   * Number of failed assert found in results.
   */
  private int assertsCount = -1;

  /**
   * Number of successful reports found in results.
   */
  private int reportsCount = -1;

  /**
   * Constructor of SchematronResult that accept the source file name (or systemID)
   *
   * @param systemID The system ID of the XML for which this result instance is built.
   */
  public SchematronResult(String systemID) {
    this.systemID = systemID;
  }

  /**
   * @return this object's systemID.
   */
  public String getSystemID() {
    return this.systemID;
  }

  /**
   * @return <code>true</code> if there's no failed assertion;
   *         <code>false</code> if there is failed assertion
   */
  public boolean isValid() {
    return this.assertsCount == 0;
  }

  /**
   * @return <code>true</code> if there's no failed assertion;
   *         <code>false</code> if there is failed assertion
   */
  public boolean hasAsserts() {
    return this.assertsCount > 0;
  }

  /**
   * @return <code>true</code> if there's no failed assertion;
   *         <code>false</code> if there is failed assertion
   */
  public boolean hasReports() {
    return this.reportsCount > 0;
  }

  /**
   * Setter for SVRL content/file.
   *
   * @param svrl The corresponding generated by SVRL.
   */
  public void setSVRL(String svrl, int assertsCount, int reportsCount) {
    this.svrl = svrl;
    this.assertsCount = assertsCount;
    this.reportsCount = reportsCount;
  }

  /**
   * Setter for SVRL content/file
   *
   * @param svrl The corresponding generated by SVRL.
   */
  @Deprecated
  public void setSVRL(String svrl) {
    this.svrl = removeXMLheader(svrl);
  }

  /**
   * Parse the SVRL output and generate the corresponding SchematronOutput instance.
   *
   * @return the corresponding SchematronOutput instance.
   * @throws SchematronException If any error occurs during parsing.
   */
  public SchematronOutput toSchematronOutput() throws SchematronException {
    return SVRLParser.parse(new StringReader(this.svrl));
  }

  /**
   * @return SVRL content as String representation.
   */
  public String getSVRLAsString() {
    return this.svrl;
  }

  /**
   * Print the failed assertion messages only on the specified stream.
   *
   * <p>This method is used only when "failOnError" is set to true.
   *
   * @param out SchematronTask object for message logging
   */
  @Deprecated
  public void printFailedMessage(PrintStream out) {
    if (this.assertsCount > 0) {
      SchematronOutput output = toSchematronOutputSilently();
      out.println("Source file: " + removePath(this.systemID));
      for (AssertOrReport failedAssertion : output.getFailedAsserts()) {
        out.println(failedAssertion.toMessageString());
      }
    }
  }

  /**
   * Return the failed assertion message only.
   *
   * <p>This method is used for Pageseeder Schematron Validation error output.
   *
   * @return a concatenation of all failed messages.
   */
  public String getFailedMessage() {
    StringBuilder out = new StringBuilder();
    if (this.assertsCount > 0) {
      SchematronOutput output = toSchematronOutputSilently();
      for (AssertOrReport failedAssertion : output.getFailedAsserts()) {
        out.append(failedAssertion.toMessageString());
      }
    }
    return out.toString();
  }

  /**
   * Print both failed assertion and successful report message to the console.
   */
  @Deprecated
  public void printAllMessage(PrintStream out) {
    if (this.assertsCount > 0 || this.reportsCount > 0) {
      if (this.systemID != null)
        out.println(("Source file: " + removePath(this.systemID)));
      SchematronOutput output = toSchematronOutputSilently();
      for (AssertOrReport assertion : output.getAllAssertsOrReports()) {
        out.println(assertion.toMessageString());
      }
    }
  }

  /**
   * @return the list of failed assertions
   */
  @Deprecated
  public List<String> getFailedAssertions() {
    SchematronOutput output = toSchematronOutputSilently();
    return SVRL.toMessageList(output.getFailedAsserts());
  }

  /**
   * @return the list of successful reports
   */
  @Deprecated
  public List<String> getSuccessfulReports() {
    SchematronOutput output = toSchematronOutputSilently();
    return SVRL.toMessageList(output.getSuccessfulReports());
  }

  // private helpers
  // ----------------------------------------------------------------------------------------------

  /**
   * Parse the SVRL content to extract any failed or success message.
   *
   * <p>The message will be stored in failedAssertions and successfulReports by SVRL handler.
   */
  private SchematronOutput toSchematronOutputSilently() {
    try {
      return SVRLParser.parse(new StringReader(this.svrl));
    } catch (SchematronException ex) {
      throw new IllegalStateException("Invalid SVRL content", ex);
    }
  }

  /**
   * Given an XML content, remove the XML header (first line if starts with <?xml)
   *
   * @param svrl The XML content of the SVRL including the XML declaration.
   *
   * @return the SVRL content without the XML header(first line) */
  private String removeXMLheader(String svrl) {
    int firstLineEnd = svrl.indexOf("\n");
    // Handle Unicode BOM
    if (svrl.startsWith("<?xml ")
     || svrl.startsWith("<?xml ", 1)
     || svrl.startsWith("<?xml ", 2)
     || svrl.startsWith("<?xml ", 3)) return svrl.substring(firstLineEnd + 1);
    else return svrl;
  }

  /**
   * Given full path to the path and return file name only
   *
   * @param filename full path to a file
   * @return name of the file only without any of the path
   */
  private String removePath(String filename) {
    String[] split = filename.split("/");
    return split[split.length - 1];
  }

}
