/*
 * Copyright (c) 1999-2016 Allette Systems Pty Ltd
 */
package org.pageseeder.flint.lucene.query;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryparser.flexible.standard.StandardQueryParser;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.suggest.document.CompletionAnalyzer;
import org.apache.lucene.search.suggest.document.PrefixCompletionQuery;
import org.pageseeder.flint.lucene.search.Fields;
import org.pageseeder.xmlwriter.XMLWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * A generic PageSeeder query object.
 *
 * @author Christophe Lauret
 * @version 26 October 2010
 */
public final class CompletionQuery implements SearchQuery {

  /** Logger for this class. */
  private static final Logger LOGGER = LoggerFactory.getLogger(CompletionQuery.class);

  /**
   * The Lucene query object to use for this search
   */
  private final Query _query;

  private final Sort _sort = Sort.RELEVANCE;

  private final List<Term> _terms;
  /**
   * Creates a new query.
   *
   * @param input            the input text
   * @param fields           the fields to search in
   * @param predicateFilter  a lucene predicate to filter the results with (can be null, will be combined with query)
   * @param filters          a lucene query to filter the results with (can be null, will be combined with predicate)
   * @param analyzer         the lucene analyzer
   */
  public CompletionQuery(String input, Collection<String> fields, String predicateFilter, Query filters, Analyzer analyzer) {
    if (input == null) throw new NullPointerException("input");
    if (fields == null) throw new NullPointerException("fields");
    CompletionAnalyzer ca = new CompletionAnalyzer(analyzer);
    this._terms = toTerms(input, fields, ca);
    if (this._terms.isEmpty()) throw new IllegalArgumentException("input,fields");
    this._query = toQuery(this._terms, filtersQuery(predicateFilter, filters, ca), ca);
  }

  /**
   * Returns how the search results should be sorted.
   *
   * @return the sort.
   */
  @Override
  public Sort getSort() {
    return this._sort;
  }

  /**
   * Indicates whether this query is <code>null</code>.
   *
   * @return <code>true</code> if the query is <code>null</code>;
   *         <code>false</code> otherwise.
   */
  @Override
  public boolean isEmpty() {
    return this._query == null;
  }

  /**
   * Returns the Lucene query instance to run on the index.
   *
   * @return the Lucene query instance to run on the index.
   */
  @Override
  public Query toQuery() {
    return this._query;
  }

  /**
   * Returns an xml representation of this query.
   *
   * @param xml The XML Writer.
   *
   * @throws IOException Should it be reported by the XML writer.
   */
  @Override
  public void toXML(XMLWriter xml) throws IOException {
    xml.openElement("completion-query", true);
    xml.attribute("empty", Boolean.toString(isEmpty()));
    if (!isEmpty()) {
      xml.attribute("query", this._query.toString());
      // facets
      for (Term t : this._terms) {
        xml.openElement("term");
        xml.attribute("field", t.field());
        xml.attribute("text", t.text());
        xml.closeElement();
      }
    }
    xml.closeElement();
  }

  // private helper
  // ----------------------------------------------------------------------------------------------

  /**
   * Build the list of terms using the input and the analyzer
   *
   * @param input     the input text
   * @param fields    the fields to search in
   * @param analyzer  the index analyzer
   *
   * @return the list of terms
   */
  private static List<Term> toTerms(String input, Collection<String> fields, Analyzer analyzer) {
    List<Term> terms = new ArrayList<>();
    for (String field : fields) {
      for (String word : Fields.toTerms(field, input, analyzer)) {
        terms.add(new Term(field, word));
      }
    }
    return terms;
  }


  /**
   * Builds the filters query.
   *
   * @param predicate  a lucene predicate
   * @param filter     a lucene query
   * @param analyzer   the analyzer
   *
   * @return the Lucene Query
   */
  private static Query filtersQuery(String predicate, Query filter, Analyzer analyzer) {
    Query predicateQuery = null;
    if (predicate != null && !predicate.isEmpty()) {
      StandardQueryParser parser = new StandardQueryParser(analyzer);
      parser.setAllowLeadingWildcard(false);
      try {
        predicateQuery = parser.parse(predicate, "pstype");
      } catch (Exception ex) {
        LOGGER.error("Unable to parse index predicate", ex);
      }
    }
    if (predicateQuery == null && filter != null) return filter;
    if (filter == null && predicateQuery != null) return predicateQuery;
    if (predicateQuery != null) {
      BooleanQuery.Builder combined = new BooleanQuery.Builder();
      combined.add(filter, Occur.MUST);
      combined.add(predicateQuery, Occur.MUST);
      return combined.build();
    }
    return null;
  }

  /**
   * Builds the query for the specified argument.
   *
   * @param terms    the list of terms
   * @param filters  a filter query
   * @param analyzer the index analyzer
   *
   * @return the Lucene Query
   */
  private static Query toQuery(List<Term> terms, Query filters, Analyzer analyzer) {
    // query
    PrefixCompletionQuery query = null;
//    for (Term t : terms) {
//      Term suggestTerm = new Term(Fields.suggestFieldName(t.field()), t.text());
//      if (query == null) {
//        query = new PrefixCompletionQuery(analyzer, suggestTerm);
//      } else {
//        query = new PrefixCompletionQuery(analyzer, suggestTerm, query.getFilter());
//      }
//    }
//    if (filters != null) {
//      BooleanQuery.Builder bq = new BooleanQuery.Builder();
//      bq.add(query, Occur.MUST);
//      bq.add(filters, Occur.MUST);
//      return bq.build();
//    }
    return query;
  }

}
