/*
 * Copyright (c) 2010-2021 Allette Systems (Australia)
 *    http://www.allette.com.au
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.pageseeder.diffx.action;

import org.jetbrains.annotations.NotNull;
import org.pageseeder.diffx.api.DiffHandler;
import org.pageseeder.diffx.api.Operator;

import java.util.ArrayList;
import java.util.List;

/**
 * Generates a list of actions from the output of the algorithms.
 * <p>
 * This handler is useful to capture the operations resulting from a diff and generate and edit script.
 *
 * @author Christophe Lauret
 * @version 0.9.0
 * @see OperationsBuffer
 */
public class ActionsBuffer<T> implements DiffHandler<T> {

  /**
   * The list of actions produced by this handler.
   */
  private final List<Action<T>> actions = new ArrayList<>();

  /**
   * The action used in the last operation.
   */
  private Action<T> action = null;

  @Override
  public void handle(@NotNull Operator operator, @NotNull T token) {
    setupAction(operator);
    this.action.add(token);
  }

  /**
   * @return the list of actions generated by this formatter.
   */
  public List<Action<T>> getActions() {
    return this.actions;
  }

  /**
   * Count the number of atomic operations of type INS or DEL.
   *
   * @return The number of edits in the buffer.
   */
  public int countEdits() {
    int edits = 0;
    for (Action<?> action : this.actions) if (action.operator().isEdit()) edits += action.tokens().size();
    return edits;
  }

  /**
   * Sets up the action prior to handling the operation.
   *
   * <p>If the action does not exist or is of a different type, create a new one
   * and add to the list of actions.
   *
   * @param operator The type of action.
   */
  private void setupAction(Operator operator) {
    if (this.action == null || this.action.operator() != operator) {
      this.action = new Action<>(operator);
      this.actions.add(this.action);
    }
  }

  /**
   * Apply the action captured by this handler to the specified handler.
   * <p>
   * This method invokes both the start and end methods on the handler.
   *
   * @param handler receives start, handler and end events.
   */
  public void applyTo(DiffHandler<T> handler) {
    handler.start();
    for (Action<T> action : this.actions) {
      for (T token : action.tokens()) {
        handler.handle(action.operator(), token);
      }
    }
    handler.end();
  }

  @Override
  public String toString() {
    return "ActionsBuffer";
  }

}
