package org.openjdk.jcstress.samples;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.openjdk.jcstress.infra.runners.TestConfig;
import org.openjdk.jcstress.infra.collectors.TestResultCollector;
import org.openjdk.jcstress.infra.runners.Runner;
import org.openjdk.jcstress.infra.runners.WorkerSync;
import org.openjdk.jcstress.util.Counter;
import org.openjdk.jcstress.vm.WhiteBoxSupport;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Callable;
import java.util.Collections;
import java.util.List;
import org.openjdk.jcstress.os.AffinitySupport;
import org.openjdk.jcstress.samples.JMMSample_02_WordTearing.BitSets;
import org.openjdk.jcstress.infra.results.ZZ_Result;

public final class JMMSample_02_WordTearing_BitSets_jcstress extends Runner<ZZ_Result> {

    volatile WorkerSync workerSync;
    BitSets[] gs;
    ZZ_Result[] gr;

    public JMMSample_02_WordTearing_BitSets_jcstress(TestConfig config, TestResultCollector collector, ExecutorService pool) {
        super(config, collector, pool, "org.openjdk.jcstress.samples.JMMSample_02_WordTearing.BitSets");
    }

    @Override
    public Counter<ZZ_Result> sanityCheck() throws Throwable {
        Counter<ZZ_Result> counter = new Counter<>();
        sanityCheck_API(counter);
        sanityCheck_Footprints(counter);
        return counter;
    }

    private void sanityCheck_API(Counter<ZZ_Result> counter) throws Throwable {
        final BitSets s = new BitSets();
        final ZZ_Result r = new ZZ_Result();
        Collection<Future<?>> res = new ArrayList<>();
        res.add(pool.submit(() -> s.writer1()));
        res.add(pool.submit(() -> s.writer2()));
        for (Future<?> f : res) {
            try {
                f.get();
            } catch (ExecutionException e) {
                throw e.getCause();
            }
        }
        try {
            pool.submit(() ->s.arbiter(r)).get();
        } catch (ExecutionException e) {
            throw e.getCause();
        }
        counter.record(r);
    }

    private void sanityCheck_Footprints(Counter<ZZ_Result> counter) throws Throwable {
        config.adjustStrides(size -> {
            BitSets[] ls = new BitSets[size];
            ZZ_Result[] lr = new ZZ_Result[size];
            workerSync = new WorkerSync(false, 2, config.spinLoopStyle);
            for (int c = 0; c < size; c++) {
                BitSets s = new BitSets();
                ZZ_Result r = new ZZ_Result();
                lr[c] = r;
                ls[c] = s;
                s.writer1();
                s.writer2();
                s.arbiter(r);
                counter.record(r);
            }
        });
    }

    @Override
    public Counter<ZZ_Result> internalRun() {
        gs = new BitSets[config.minStride];
        gr = new ZZ_Result[config.minStride];
        for (int c = 0; c < config.minStride; c++) {
            gs[c] = new BitSets();
            gr[c] = new ZZ_Result();
        }
        workerSync = new WorkerSync(false, 2, config.spinLoopStyle);

        control.isStopped = false;

        List<Callable<Counter<ZZ_Result>>> tasks = new ArrayList<>();
        tasks.add(this::task_writer1);
        tasks.add(this::task_writer2);
        Collections.shuffle(tasks);

        Collection<Future<Counter<ZZ_Result>>> results = new ArrayList<>();
        for (Callable<Counter<ZZ_Result>> task : tasks) {
            results.add(pool.submit(task));
        }

        if (config.time > 0) {
            try {
                TimeUnit.MILLISECONDS.sleep(config.time);
            } catch (InterruptedException e) {
            }
        }

        control.isStopped = true;

        waitFor(results);

        Counter<ZZ_Result> counter = new Counter<>();
        for (Future<Counter<ZZ_Result>> f : results) {
            try {
                counter.merge(f.get());
            } catch (Throwable e) {
                throw new IllegalStateException(e);
            }
        }
        return counter;
    }

    private void jcstress_consume(Counter<ZZ_Result> cnt, int a) {
        BitSets[] ls = gs;
        ZZ_Result[] lr = gr;
        int len = ls.length;
        int left = a * len / 2;
        int right = (a + 1) * len / 2;
        for (int c = left; c < right; c++) {
            ZZ_Result r = lr[c];
            BitSets s = ls[c];
            s.arbiter(r);
            ls[c] = new BitSets();
            cnt.record(r);
            r.r1 = false;
            r.r2 = false;
        }
    }

    private void jcstress_update(WorkerSync sync) {
        BitSets[] ls = gs;
        ZZ_Result[] lr = gr;
        int len = ls.length;

        int newLen = sync.updateStride ? Math.max(config.minStride, Math.min(len * 2, config.maxStride)) : len;

        if (newLen > len) {
            ls = Arrays.copyOf(ls, newLen);
            lr = Arrays.copyOf(lr, newLen);
            for (int c = len; c < newLen; c++) {
                ls[c] = new BitSets();
                lr[c] = new ZZ_Result();
            }
            gs = ls;
            gr = lr;
         }

        workerSync = new WorkerSync(control.isStopped, 2, config.spinLoopStyle);
   }
    private void jcstress_sink(int v) {};
    private void jcstress_sink(short v) {};
    private void jcstress_sink(byte v) {};
    private void jcstress_sink(char v) {};
    private void jcstress_sink(long v) {};
    private void jcstress_sink(float v) {};
    private void jcstress_sink(double v) {};
    private void jcstress_sink(Object v) {};

    private Counter<ZZ_Result> task_writer1() {
        Counter<ZZ_Result> counter = new Counter<>();
        AffinitySupport.bind(config.cpuMap.actorMap()[0]);
        while (true) {
            WorkerSync sync = workerSync;
            if (sync.stopped) {
                return counter;
            }
            sync.preRun();
            run_writer1(gs, gr);
            sync.postRun();
            jcstress_consume(counter, 0);
            if (sync.tryStartUpdate()) {
                jcstress_update(sync);
            }
            sync.postUpdate();
        }
    }

    private void run_writer1(BitSets[] gs, ZZ_Result[] gr) {
        BitSets[] ls = gs;
        ZZ_Result[] lr = gr;
        int size = ls.length;
        for (int c = 0; c < size; c++) {
            BitSets s = ls[c];
            s.writer1();
        }
    }

    private Counter<ZZ_Result> task_writer2() {
        Counter<ZZ_Result> counter = new Counter<>();
        AffinitySupport.bind(config.cpuMap.actorMap()[1]);
        while (true) {
            WorkerSync sync = workerSync;
            if (sync.stopped) {
                return counter;
            }
            sync.preRun();
            run_writer2(gs, gr);
            sync.postRun();
            jcstress_consume(counter, 1);
            if (sync.tryStartUpdate()) {
                jcstress_update(sync);
            }
            sync.postUpdate();
        }
    }

    private void run_writer2(BitSets[] gs, ZZ_Result[] gr) {
        BitSets[] ls = gs;
        ZZ_Result[] lr = gr;
        int size = ls.length;
        for (int c = 0; c < size; c++) {
            BitSets s = ls[c];
            s.writer2();
        }
    }

}
