package org.openjdk.jcstress.samples;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.openjdk.jcstress.infra.runners.TestConfig;
import org.openjdk.jcstress.infra.collectors.TestResultCollector;
import org.openjdk.jcstress.infra.runners.Runner;
import org.openjdk.jcstress.infra.runners.WorkerSync;
import org.openjdk.jcstress.util.Counter;
import org.openjdk.jcstress.vm.WhiteBoxSupport;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Callable;
import java.util.Collections;
import java.util.List;
import org.openjdk.jcstress.os.AffinitySupport;
import org.openjdk.jcstress.samples.JMMSample_01_AccessAtomicity.VolatileLongs;
import org.openjdk.jcstress.infra.results.J_Result;

public final class JMMSample_01_AccessAtomicity_VolatileLongs_jcstress extends Runner<J_Result> {

    volatile WorkerSync workerSync;
    VolatileLongs[] gs;
    J_Result[] gr;

    public JMMSample_01_AccessAtomicity_VolatileLongs_jcstress(TestConfig config, TestResultCollector collector, ExecutorService pool) {
        super(config, collector, pool, "org.openjdk.jcstress.samples.JMMSample_01_AccessAtomicity.VolatileLongs");
    }

    @Override
    public Counter<J_Result> sanityCheck() throws Throwable {
        Counter<J_Result> counter = new Counter<>();
        sanityCheck_API(counter);
        sanityCheck_Footprints(counter);
        return counter;
    }

    private void sanityCheck_API(Counter<J_Result> counter) throws Throwable {
        final VolatileLongs s = new VolatileLongs();
        final J_Result r = new J_Result();
        Collection<Future<?>> res = new ArrayList<>();
        res.add(pool.submit(() -> s.writer()));
        res.add(pool.submit(() -> s.reader(r)));
        for (Future<?> f : res) {
            try {
                f.get();
            } catch (ExecutionException e) {
                throw e.getCause();
            }
        }
        counter.record(r);
    }

    private void sanityCheck_Footprints(Counter<J_Result> counter) throws Throwable {
        config.adjustStrides(size -> {
            VolatileLongs[] ls = new VolatileLongs[size];
            J_Result[] lr = new J_Result[size];
            workerSync = new WorkerSync(false, 2, config.spinLoopStyle);
            for (int c = 0; c < size; c++) {
                VolatileLongs s = new VolatileLongs();
                J_Result r = new J_Result();
                lr[c] = r;
                ls[c] = s;
                s.writer();
                s.reader(r);
                counter.record(r);
            }
        });
    }

    @Override
    public Counter<J_Result> internalRun() {
        gs = new VolatileLongs[config.minStride];
        gr = new J_Result[config.minStride];
        for (int c = 0; c < config.minStride; c++) {
            gs[c] = new VolatileLongs();
            gr[c] = new J_Result();
        }
        workerSync = new WorkerSync(false, 2, config.spinLoopStyle);

        control.isStopped = false;

        List<Callable<Counter<J_Result>>> tasks = new ArrayList<>();
        tasks.add(this::task_writer);
        tasks.add(this::task_reader);
        Collections.shuffle(tasks);

        Collection<Future<Counter<J_Result>>> results = new ArrayList<>();
        for (Callable<Counter<J_Result>> task : tasks) {
            results.add(pool.submit(task));
        }

        if (config.time > 0) {
            try {
                TimeUnit.MILLISECONDS.sleep(config.time);
            } catch (InterruptedException e) {
            }
        }

        control.isStopped = true;

        waitFor(results);

        Counter<J_Result> counter = new Counter<>();
        for (Future<Counter<J_Result>> f : results) {
            try {
                counter.merge(f.get());
            } catch (Throwable e) {
                throw new IllegalStateException(e);
            }
        }
        return counter;
    }

    private void jcstress_consume(Counter<J_Result> cnt, int a) {
        VolatileLongs[] ls = gs;
        J_Result[] lr = gr;
        int len = ls.length;
        int left = a * len / 2;
        int right = (a + 1) * len / 2;
        for (int c = left; c < right; c++) {
            J_Result r = lr[c];
            VolatileLongs s = ls[c];
            s.v = 0;
            cnt.record(r);
            r.r1 = 0;
        }
    }

    private void jcstress_update(WorkerSync sync) {
        VolatileLongs[] ls = gs;
        J_Result[] lr = gr;
        int len = ls.length;

        int newLen = sync.updateStride ? Math.max(config.minStride, Math.min(len * 2, config.maxStride)) : len;

        if (newLen > len) {
            ls = Arrays.copyOf(ls, newLen);
            lr = Arrays.copyOf(lr, newLen);
            for (int c = len; c < newLen; c++) {
                ls[c] = new VolatileLongs();
                lr[c] = new J_Result();
            }
            gs = ls;
            gr = lr;
         }

        workerSync = new WorkerSync(control.isStopped, 2, config.spinLoopStyle);
   }
    private void jcstress_sink(int v) {};
    private void jcstress_sink(short v) {};
    private void jcstress_sink(byte v) {};
    private void jcstress_sink(char v) {};
    private void jcstress_sink(long v) {};
    private void jcstress_sink(float v) {};
    private void jcstress_sink(double v) {};
    private void jcstress_sink(Object v) {};

    private Counter<J_Result> task_writer() {
        Counter<J_Result> counter = new Counter<>();
        AffinitySupport.bind(config.cpuMap.actorMap()[0]);
        while (true) {
            WorkerSync sync = workerSync;
            if (sync.stopped) {
                return counter;
            }
            sync.preRun();
            run_writer(gs, gr);
            sync.postRun();
            jcstress_consume(counter, 0);
            if (sync.tryStartUpdate()) {
                jcstress_update(sync);
            }
            sync.postUpdate();
        }
    }

    private void run_writer(VolatileLongs[] gs, J_Result[] gr) {
        VolatileLongs[] ls = gs;
        J_Result[] lr = gr;
        int size = ls.length;
        for (int c = 0; c < size; c++) {
            VolatileLongs s = ls[c];
            s.writer();
        }
    }

    private Counter<J_Result> task_reader() {
        Counter<J_Result> counter = new Counter<>();
        AffinitySupport.bind(config.cpuMap.actorMap()[1]);
        while (true) {
            WorkerSync sync = workerSync;
            if (sync.stopped) {
                return counter;
            }
            sync.preRun();
            run_reader(gs, gr);
            sync.postRun();
            jcstress_consume(counter, 1);
            if (sync.tryStartUpdate()) {
                jcstress_update(sync);
            }
            sync.postUpdate();
        }
    }

    private void run_reader(VolatileLongs[] gs, J_Result[] gr) {
        VolatileLongs[] ls = gs;
        J_Result[] lr = gr;
        int size = ls.length;
        for (int c = 0; c < size; c++) {
            VolatileLongs s = ls[c];
            J_Result r = lr[c];
            jcstress_sink(r.jcstress_trap);
            s.reader(r);
        }
    }

}
