package org.openjdk.jcstress.samples;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.openjdk.jcstress.infra.runners.TestConfig;
import org.openjdk.jcstress.infra.collectors.TestResultCollector;
import org.openjdk.jcstress.infra.runners.Runner;
import org.openjdk.jcstress.infra.runners.WorkerSync;
import org.openjdk.jcstress.util.Counter;
import org.openjdk.jcstress.vm.WhiteBoxSupport;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Callable;
import java.util.Collections;
import java.util.List;
import org.openjdk.jcstress.os.AffinitySupport;
import org.openjdk.jcstress.samples.ConcurrencySample_01_OperationAtomicity.PlainIncrement;
import org.openjdk.jcstress.infra.results.I_Result;

public final class ConcurrencySample_01_OperationAtomicity_PlainIncrement_jcstress extends Runner<I_Result> {

    volatile WorkerSync workerSync;
    PlainIncrement[] gs;
    I_Result[] gr;

    public ConcurrencySample_01_OperationAtomicity_PlainIncrement_jcstress(TestConfig config, TestResultCollector collector, ExecutorService pool) {
        super(config, collector, pool, "org.openjdk.jcstress.samples.ConcurrencySample_01_OperationAtomicity.PlainIncrement");
    }

    @Override
    public Counter<I_Result> sanityCheck() throws Throwable {
        Counter<I_Result> counter = new Counter<>();
        sanityCheck_API(counter);
        sanityCheck_Footprints(counter);
        return counter;
    }

    private void sanityCheck_API(Counter<I_Result> counter) throws Throwable {
        final PlainIncrement s = new PlainIncrement();
        final I_Result r = new I_Result();
        Collection<Future<?>> res = new ArrayList<>();
        res.add(pool.submit(() -> s.actor1()));
        res.add(pool.submit(() -> s.actor2()));
        for (Future<?> f : res) {
            try {
                f.get();
            } catch (ExecutionException e) {
                throw e.getCause();
            }
        }
        try {
            pool.submit(() ->s.arbiter(r)).get();
        } catch (ExecutionException e) {
            throw e.getCause();
        }
        counter.record(r);
    }

    private void sanityCheck_Footprints(Counter<I_Result> counter) throws Throwable {
        config.adjustStrides(size -> {
            PlainIncrement[] ls = new PlainIncrement[size];
            I_Result[] lr = new I_Result[size];
            workerSync = new WorkerSync(false, 2, config.spinLoopStyle);
            for (int c = 0; c < size; c++) {
                PlainIncrement s = new PlainIncrement();
                I_Result r = new I_Result();
                lr[c] = r;
                ls[c] = s;
                s.actor1();
                s.actor2();
                s.arbiter(r);
                counter.record(r);
            }
        });
    }

    @Override
    public Counter<I_Result> internalRun() {
        gs = new PlainIncrement[config.minStride];
        gr = new I_Result[config.minStride];
        for (int c = 0; c < config.minStride; c++) {
            gs[c] = new PlainIncrement();
            gr[c] = new I_Result();
        }
        workerSync = new WorkerSync(false, 2, config.spinLoopStyle);

        control.isStopped = false;

        List<Callable<Counter<I_Result>>> tasks = new ArrayList<>();
        tasks.add(this::task_actor1);
        tasks.add(this::task_actor2);
        Collections.shuffle(tasks);

        Collection<Future<Counter<I_Result>>> results = new ArrayList<>();
        for (Callable<Counter<I_Result>> task : tasks) {
            results.add(pool.submit(task));
        }

        if (config.time > 0) {
            try {
                TimeUnit.MILLISECONDS.sleep(config.time);
            } catch (InterruptedException e) {
            }
        }

        control.isStopped = true;

        waitFor(results);

        Counter<I_Result> counter = new Counter<>();
        for (Future<Counter<I_Result>> f : results) {
            try {
                counter.merge(f.get());
            } catch (Throwable e) {
                throw new IllegalStateException(e);
            }
        }
        return counter;
    }

    private void jcstress_consume(Counter<I_Result> cnt, int a) {
        PlainIncrement[] ls = gs;
        I_Result[] lr = gr;
        int len = ls.length;
        int left = a * len / 2;
        int right = (a + 1) * len / 2;
        for (int c = left; c < right; c++) {
            I_Result r = lr[c];
            PlainIncrement s = ls[c];
            s.arbiter(r);
            s.v = 0;
            cnt.record(r);
            r.r1 = 0;
        }
    }

    private void jcstress_update(WorkerSync sync) {
        PlainIncrement[] ls = gs;
        I_Result[] lr = gr;
        int len = ls.length;

        int newLen = sync.updateStride ? Math.max(config.minStride, Math.min(len * 2, config.maxStride)) : len;

        if (newLen > len) {
            ls = Arrays.copyOf(ls, newLen);
            lr = Arrays.copyOf(lr, newLen);
            for (int c = len; c < newLen; c++) {
                ls[c] = new PlainIncrement();
                lr[c] = new I_Result();
            }
            gs = ls;
            gr = lr;
         }

        workerSync = new WorkerSync(control.isStopped, 2, config.spinLoopStyle);
   }
    private void jcstress_sink(int v) {};
    private void jcstress_sink(short v) {};
    private void jcstress_sink(byte v) {};
    private void jcstress_sink(char v) {};
    private void jcstress_sink(long v) {};
    private void jcstress_sink(float v) {};
    private void jcstress_sink(double v) {};
    private void jcstress_sink(Object v) {};

    private Counter<I_Result> task_actor1() {
        Counter<I_Result> counter = new Counter<>();
        AffinitySupport.bind(config.cpuMap.actorMap()[0]);
        while (true) {
            WorkerSync sync = workerSync;
            if (sync.stopped) {
                return counter;
            }
            sync.preRun();
            run_actor1(gs, gr);
            sync.postRun();
            jcstress_consume(counter, 0);
            if (sync.tryStartUpdate()) {
                jcstress_update(sync);
            }
            sync.postUpdate();
        }
    }

    private void run_actor1(PlainIncrement[] gs, I_Result[] gr) {
        PlainIncrement[] ls = gs;
        I_Result[] lr = gr;
        int size = ls.length;
        for (int c = 0; c < size; c++) {
            PlainIncrement s = ls[c];
            s.actor1();
        }
    }

    private Counter<I_Result> task_actor2() {
        Counter<I_Result> counter = new Counter<>();
        AffinitySupport.bind(config.cpuMap.actorMap()[1]);
        while (true) {
            WorkerSync sync = workerSync;
            if (sync.stopped) {
                return counter;
            }
            sync.preRun();
            run_actor2(gs, gr);
            sync.postRun();
            jcstress_consume(counter, 1);
            if (sync.tryStartUpdate()) {
                jcstress_update(sync);
            }
            sync.postUpdate();
        }
    }

    private void run_actor2(PlainIncrement[] gs, I_Result[] gr) {
        PlainIncrement[] ls = gs;
        I_Result[] lr = gr;
        int size = ls.length;
        for (int c = 0; c < size; c++) {
            PlainIncrement s = ls[c];
            s.actor2();
        }
    }

}
