/*
 * Copyright (c) 2016-2018 "Neo4j, Inc." [https://neo4j.com]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Attribution Notice under the terms of the Apache License 2.0
 *
 * This work was created by the collective efforts of the openCypher community.
 * Without limiting the terms of Section 6, any Derivative Work that is not
 * approved by the public consensus process of the openCypher Implementers Group
 * should not be described as “Cypher” (and Cypher® is a registered trademark of
 * Neo4j Inc.) or as "openCypher". Extensions by implementers or prototypes or
 * proposals for change that have been documented or implemented should only be
 * described as "implementation extensions to Cypher" or as "proposed changes to
 * Cypher that are not yet approved by the openCypher community".
 */
package org.opencypher.spark.impl.util

import org.opencypher.spark.api.io.{Labels, Node, Relationship, RelationshipType}

import scala.annotation.StaticAnnotation
import scala.reflect.runtime.universe._
import scala.tools.reflect.ToolBox

private[spark] object Annotation {
  def labels[E <: Node: TypeTag]: Set[String] = synchronized {
    get[Labels, E] match {
      case Some(ls) => ls.labels.toSet
      case None     => Set(runtimeClass[E].getSimpleName)
    }
  }

  def relType[E <: Relationship: TypeTag]: String = synchronized {
    get[RelationshipType, E] match {
      case Some(RelationshipType(tpe)) => tpe
      case None                        => runtimeClass[E].getSimpleName.toUpperCase
    }
  }

  def get[A <: StaticAnnotation: TypeTag, E: TypeTag]: Option[A] = synchronized {
    val maybeAnnotation = staticClass[E].annotations.find(_.tree.tpe =:= typeOf[A])
    maybeAnnotation.map { annotation =>
      val tb = typeTag[E].mirror.mkToolBox()
      val instance = tb.eval(tb.untypecheck(annotation.tree)).asInstanceOf[A]
      instance
    }
  }

  private def runtimeClass[E: TypeTag]: Class[E] = synchronized {
    val tag = typeTag[E]
    val mirror = tag.mirror
    val runtimeClass = mirror.runtimeClass(tag.tpe.typeSymbol.asClass)
    runtimeClass.asInstanceOf[Class[E]]
  }

  private def staticClass[E: TypeTag]: ClassSymbol = synchronized {
    val mirror = typeTag[E].mirror
    mirror.staticClass(runtimeClass[E].getCanonicalName)
  }
}
