package org.krproject.ocean.vitamins.param.service.inner.provided;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.function.Function;

import org.krproject.ocean.vitamins.param.domain.ParamAuditEntity;
import org.krproject.ocean.vitamins.param.domain.ParamAuditRepository;
import org.krproject.ocean.vitamins.param.domain.ParamObjectEntity;
import org.krproject.ocean.vitamins.param.domain.ParamObjectRepository;
import org.krproject.ocean.vitamins.param.domain.enums.ParamOperationEnum;
import org.krproject.ocean.vitamins.param.exception.ErrorCode;
import org.krproject.ocean.vitamins.param.exception.ParamException;
import org.krproject.ocean.vitamins.param.service.inner.ParamStoreService;
import org.krproject.ocean.vitamins.param.service.inner.model.ParamModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

import lombok.extern.slf4j.Slf4j;

/**
 * 参数存储服务-数据库媒介服务实现.
 * @author zhongyang
 *
 */
@Slf4j
public class ParamStoreServiceDatabaseImpl implements ParamStoreService {

	private static final ObjectMapper objectMapper = new ObjectMapper();
	
	static {
		objectMapper.configure(SerializationFeature.WRAP_ROOT_VALUE, true);
		objectMapper.configure(DeserializationFeature.UNWRAP_ROOT_VALUE, true);
		objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
	}
	
	@Autowired
	private ParamObjectRepository paramObjectRepository;

	@Autowired
	private ParamAuditRepository paramAuditRepository;

	
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public Integer countParamStore(String orgId, String paramClass) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		
		return this.paramObjectRepository.countByOrgIdAndParamClass(orgId, paramClass);
	}
	
	
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public List<ParamModel> listParamStore(String orgId, String paramClass) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		
		List<ParamObjectEntity> entityList = this.paramObjectRepository.findByOrgIdAndParamClass(orgId, paramClass);
		if (entityList != null && entityList.size() > 0) {
			List<ParamModel> paramModels = new ArrayList<ParamModel>();
			for (ParamObjectEntity entity : entityList) {
				paramModels.add(new ParamStoreConverter().apply(entity));				
			}
			return paramModels;
		}
		return null;
	}
	
	
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public ParamModel getParamStore(String orgId, String paramClass, String paramKey) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramKey, "指定参数key不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		
		ParamObjectEntity entity = this.paramObjectRepository.findByOrgIdAndParamClassAndParamKey(orgId, paramClass, paramKey);
		return new ParamStoreConverter().apply(entity);
	}
	
	
	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public void addParamStore(ParamModel paramModel) {
		Assert.notNull(paramModel, "指定参数实体不能为空");
		Assert.notNull(paramModel.getOrgId(), "指定参数orgId不能为空");
		Assert.notNull(paramModel.getParamKey(), "指定参数key不能为空");
		Assert.notNull(paramModel.getParamClass(), "指定参数paramClass不能为空");
		Assert.notNull(paramModel.getParamObject(), "指定参数定义不能为空");
		
		// 参数查重
		ParamObjectEntity paramObjectChk = this.paramObjectRepository.findByOrgIdAndParamClassAndParamKey(
				paramModel.getOrgId(), paramModel.getParamClass(), paramModel.getParamKey());
		if (paramObjectChk != null) {
			throw new ParamException(ErrorCode.PARAM_ALREADY_EXISTS, "参数:[" + paramModel.toString() + "],已存在!");
		}
		
		ParamObjectEntity entity = new ParamObjectEntity();
		entity.setOrgId(paramModel.getOrgId());
		entity.setParamClass(paramModel.getParamClass());
		entity.setParamKey(paramModel.getParamKey());
		entity.setParamObject(marshalParamObject(paramModel.getParamObject()));
		entity.setMtnUser(paramModel.getMtnUser());
		entity.setLastUpdateTime(paramModel.getLastUpdateTime());
		entity.setCreateTime(new Date());
		this.paramObjectRepository.saveAndFlush(entity);
		
		//记录操作日志
		ParamAuditEntity paramAuditEntity = new ParamAuditEntity();
		paramAuditEntity.setOrgId(entity.getOrgId());
		paramAuditEntity.setParamClass(entity.getParamClass());
		paramAuditEntity.setParamKey(entity.getParamKey());
		paramAuditEntity.setParamOperation(ParamOperationEnum.INSERT);
		paramAuditEntity.setMtnUser(entity.getMtnUser());
		paramAuditEntity.setMtnMemo("新增参数");
		paramAuditEntity.setOldObject(null);
		paramAuditEntity.setNewObject(entity.getParamObject());
		paramAuditEntity.setCreateTime(new Date());
		this.paramAuditRepository.saveAndFlush(paramAuditEntity);
	}
	

	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public void updateParamStore(ParamModel paramModel) {
		Assert.notNull(paramModel, "指定参数实体不能为空");
		Assert.notNull(paramModel.getOrgId(), "指定参数orgId不能为空");
		Assert.notNull(paramModel.getParamKey(), "指定参数key不能为空");
		Assert.notNull(paramModel.getParamClass(), "指定参数paramClass不能为空");
		Assert.notNull(paramModel.getParamObject(), "指定参数定义不能为空");
		
		// 查找参数
		ParamObjectEntity entity = this.paramObjectRepository.findByOrgIdAndParamClassAndParamKey(
				paramModel.getOrgId(), paramModel.getParamClass(), paramModel.getParamKey());
		if (entity == null) {
			throw new ParamException(ErrorCode.PARAM_NOT_FOUND, "参数:[" + paramModel.toString() + "],未找到!");
		}
		// 更新参数
		String oldParamObject = entity.getParamObject();
		String newParamObject = marshalParamObject(paramModel.getParamObject());
		entity.setParamObject(newParamObject);
		entity.setMtnUser(paramModel.getMtnUser());
		entity.setLastUpdateTime(paramModel.getLastUpdateTime());
		this.paramObjectRepository.saveAndFlush(entity);
		
		//记录操作日志
		ParamAuditEntity paramAuditEntity = new ParamAuditEntity();
		paramAuditEntity.setOrgId(entity.getOrgId());
		paramAuditEntity.setParamClass(entity.getParamClass());
		paramAuditEntity.setParamKey(entity.getParamKey());
		paramAuditEntity.setParamOperation(ParamOperationEnum.UPDATE);
		paramAuditEntity.setMtnUser(entity.getMtnUser());
		paramAuditEntity.setMtnMemo("更新参数");
		paramAuditEntity.setOldObject(oldParamObject);
		paramAuditEntity.setNewObject(newParamObject);
		paramAuditEntity.setCreateTime(new Date());
		this.paramAuditRepository.saveAndFlush(paramAuditEntity);
	}
	

	@Override
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public void removeParamStore(String orgId, String paramClass, String paramKey, String mtnUser) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramKey, "指定参数key不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		
		// 查找并删除参数
		ParamObjectEntity entity = this.paramObjectRepository.findByOrgIdAndParamClassAndParamKey(orgId, paramClass, paramKey);
		if (entity == null) {
			throw new ParamException(ErrorCode.PARAM_NOT_FOUND, "机构号:[" + orgId + "],ParamClass:["
					+ paramClass + "],ParamKey:[" 
					+ paramKey + "],参数未找到！");
		}
		this.paramObjectRepository.delete(entity);
		
		// 记录操作日志
		ParamAuditEntity paramAuditEntity = new ParamAuditEntity();
		paramAuditEntity.setOrgId(entity.getOrgId());
		paramAuditEntity.setParamClass(entity.getParamClass());
		paramAuditEntity.setParamKey(entity.getParamKey());
		paramAuditEntity.setParamOperation(ParamOperationEnum.DELETE);
		paramAuditEntity.setMtnUser(entity.getMtnUser());
		paramAuditEntity.setMtnMemo("删除参数");
		paramAuditEntity.setOldObject(entity.getParamObject());
		paramAuditEntity.setNewObject(null);
		paramAuditEntity.setCreateTime(new Date());
		this.paramAuditRepository.saveAndFlush(paramAuditEntity);
	}


	/**
	 * 序列化参数对象.
	 * @param paramObject 参数对象
	 * @return 参数对象字符串
	 */
	private String marshalParamObject(Object paramObject) {
		String str = null;
		try {
			str = objectMapper.writeValueAsString(paramObject);
		} catch (Exception e) {
			log.error("marshal object:{} failed with exception:{}", paramObject, e);
			throw new ParamException(ErrorCode.PARAM_PARSE_ERROR, "marshal paramObject failed!");
		}
		return str;
	}


	/**
	 * 反序列化参数对象.
	 * @param paramObjectStr 参数对象字符串
	 * @param paramObjectClz 参数对象类名
	 * @return 参数对象
	 */
	private Object unmarshalParamObject(String paramObjectStr, String paramObjectClz) {
		Object obj = null;
		try {
			Class<?> clazz = Class.forName(paramObjectClz);
			obj = objectMapper.readValue(paramObjectStr, clazz);
		} catch (Exception e) {
			log.error("unmarshal message:{} class:{} failed with exception:{}", paramObjectStr, paramObjectClz, e);
			throw new ParamException(ErrorCode.PARAM_PARSE_ERROR, "unmarshal paramObject failed!");
		}
		return obj;
	}
	
	
	/**
	 * 转换参数存储实体为参数存储模型.
	 */
	private class ParamStoreConverter implements Function<ParamObjectEntity, ParamModel> {

		@Override
		public ParamModel apply(ParamObjectEntity entity) {
			if (entity != null) {
				ParamModel model = new ParamModel();
				model.setOrgId(entity.getOrgId());
				model.setParamClass(entity.getParamClass());
				model.setParamKey(entity.getParamKey());
				model.setParamObject(unmarshalParamObject(entity.getParamObject(), entity.getParamClass()));
				model.setMtnUser(entity.getMtnUser());
				model.setLastUpdateTime(entity.getLastUpdateTime());
				return model;
			}
			return null;
		}
	}

}
