package org.krproject.ocean.vitamins.param.service.inner.provided;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.PostConstruct;

import org.krproject.ocean.vitamins.param.config.ParamProperties;
import org.krproject.ocean.vitamins.param.service.inner.ParamAccessService;
import org.krproject.ocean.vitamins.param.service.inner.ParamStoreService;
import org.krproject.ocean.vitamins.param.service.inner.model.ParamModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.SmartLifecycle;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * 参数访问服务-缓存访问实现.
 * @author zhongyang
 *
 */
@Slf4j
public class ParamAccessServiceCacheImpl implements ParamAccessService, SmartLifecycle {

	// 参数缓存
	private final ConcurrentHashMap<String, ParamModelWrapper> paramCacheMap = new ConcurrentHashMap<>();

	@Autowired
	private ParamProperties paramProperties;

	private AtomicBoolean running = new AtomicBoolean(false);
	
	private ThreadPoolTaskScheduler paramCacheCheckTaskScheduler;
	
	private ScheduledFuture<?> watchFuture;

	@PostConstruct
	public void init() {
		this.paramCacheCheckTaskScheduler = new ThreadPoolTaskScheduler();
		this.paramCacheCheckTaskScheduler.setBeanName("paramCacheCheckTaskScheduler");
		this.paramCacheCheckTaskScheduler.setThreadNamePrefix("param-cache-check-");
		this.paramCacheCheckTaskScheduler.initialize();
	}
	
	@Override
	public void start() {
		if (this.running.compareAndSet(false, true)) {
			this.watchFuture = this.paramCacheCheckTaskScheduler.scheduleWithFixedDelay(
					this::checkParamCache, this.paramProperties.getCacheCheckSeconds() * 1000);
		}
	}

	@Override
	public void stop() {
		if (this.running.compareAndSet(true, false) && this.watchFuture != null) {
			this.watchFuture.cancel(true);
		}
	}

	@Override
	public boolean isRunning() {
		return this.running.get();
	}
	
	@Override
	public int getPhase() {
		return 0;
	}

	/**
	 * 检查参数缓存与存储是否一致.
	 */
	private void checkParamCache() {
		if (this.running.get()) {
			log.debug("checking param cache:{}", this.paramCacheMap);
			
			for (Entry<String, ParamModelWrapper> entry : this.paramCacheMap.entrySet()) { 
				// 如果包含全部，则先校验总数
				ParamModelWrapper paramModelWrapper = entry.getValue();
				ParamStoreService paramStoreService = paramModelWrapper.getParamStoreService();
				if (paramModelWrapper.getContainAll()) { 
					int paramCacheSize = paramModelWrapper.getParamModelMap().size();
					int paramStoreSize = paramStoreService.countParamStore(paramModelWrapper.getOrgId(), paramModelWrapper.getParamClass());
					if (paramCacheSize != paramStoreSize) {
						log.warn("Check Param Cache Size Diff, paramType:{}, paramCacheSize:{} paramStoreSize:{}", 
								entry.getKey(), paramCacheSize, paramStoreSize);
						paramModelWrapper.getParamModelMap().clear();
						paramModelWrapper.setContainAll(false);
					}
				}
				
				// 遍历所有缓存参数，比对最后修改时间
				for (Entry<String, ParamModel> cacheEntry : paramModelWrapper.getParamModelMap().entrySet()) {
					ParamModel paramCacheModel = cacheEntry.getValue();
					ParamModel paramStoreModel = paramStoreService.getParamStore(paramCacheModel.getOrgId(), paramCacheModel.getParamClass(), paramCacheModel.getParamKey());
					if (paramStoreModel == null || paramStoreModel.getLastUpdateTime() == null || paramCacheModel.getLastUpdateTime() == null ||  
							paramStoreModel.getLastUpdateTime().compareTo(paramCacheModel.getLastUpdateTime()) != 0) { 
						log.warn("Check Param Cache Value Diff, paramKey:{}. paramCacheModel:{} paramStoreModel:{}", 
								cacheEntry.getKey(), paramCacheModel, paramStoreModel);
						paramModelWrapper.getParamModelMap().remove(cacheEntry.getKey());
						paramModelWrapper.setContainAll(false);
					}
				}
			}
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T> T getParameter(String orgId, String paramClass, String key, ParamStoreService paramStoreService) {
		log.debug("getParameter with orgId:{}, paramClass:{}, key:{}", orgId, paramClass, key);
		
		// 参数类型Key
		String paramTypeKey = getCacheTypeKey(orgId, paramClass);
		ParamModelWrapper paramModelWrapper = this.paramCacheMap.get(paramTypeKey);
		if (paramModelWrapper == null) {
			paramModelWrapper = new ParamModelWrapper();
			paramModelWrapper.setOrgId(orgId);
			paramModelWrapper.setParamClass(paramClass);
			paramModelWrapper.setParamModelMap(new ConcurrentHashMap<>());
			paramModelWrapper.setParamStoreService(paramStoreService);
			paramModelWrapper.setContainAll(false);
			this.paramCacheMap.put(paramTypeKey, paramModelWrapper);
		}
		
		ConcurrentHashMap<String, ParamModel> paramModelMap = paramModelWrapper.getParamModelMap();
		if (paramModelMap.get(key) == null) {
			log.debug("getParameter through store with orgId:{}, paramClass:{}, key:{}", orgId, paramClass, key);
			ParamModel paramModel = paramStoreService.getParamStore(orgId, paramClass, key);
			if (paramModel != null) {
				paramModelMap.put(paramModel.getParamKey(), paramModel);
			}
		}
		
		if (paramModelMap.get(key) != null) {
			return (T) paramModelMap.get(key).getParamObject();
		}
		return null;
	}


	@SuppressWarnings("unchecked")
	@Override
	public <T> Map<String, T> getParameterMap(String orgId, String paramClass, ParamStoreService paramStoreService) {
		log.debug("getParameterMap with orgId:{}, paramClass:{}", orgId, paramClass);
		
		// 参数类型Key
		String paramTypeKey = getCacheTypeKey(orgId, paramClass);
		ParamModelWrapper paramModelWrapper = this.paramCacheMap.get(paramTypeKey);
		if (paramModelWrapper == null) {
			paramModelWrapper = new ParamModelWrapper();
			paramModelWrapper.setOrgId(orgId);
			paramModelWrapper.setParamClass(paramClass);
			paramModelWrapper.setParamModelMap(new ConcurrentHashMap<>());
			paramModelWrapper.setParamStoreService(paramStoreService);
			paramModelWrapper.setContainAll(false);
			this.paramCacheMap.put(paramTypeKey, paramModelWrapper);
		}
				
		// 判断参数访问类型
		ConcurrentHashMap<String, ParamModel> paramModelMap = paramModelWrapper.getParamModelMap();
		if (!paramModelWrapper.getContainAll()) {
			log.debug("getParameterMap through store with orgId:{}, paramClass:{}", orgId, paramClass);
			List<ParamModel> paramModelList = paramStoreService.listParamStore(orgId, paramClass);
			if (paramModelList != null && paramModelList.size() > 0) {
				for (ParamModel paramModel : paramModelList) {
					paramModelMap.put(paramModel.getParamKey(), paramModel);
				}
				// 更新为已全部获取
				paramModelWrapper.setContainAll(true);
			}
		} 
		
		Map<String, T> paramMap = new HashMap<String, T>();
		for (ParamModel paramModel: paramModelMap.values()) {
			paramMap.put(paramModel.getParamKey(), (T) paramModel.getParamObject());
		}
		return paramMap;
	}
	
	
	@Override
	public void onChangedParameter(String orgId, String paramClassName, String key) {
		log.debug("onChangedParameter with orgId:{}, paramClass:{}, key:{}", orgId, paramClassName, key);

		// 参数类型Key
		String paramTypeKey = getCacheTypeKey(orgId, paramClassName);
		ParamModelWrapper paramModelWrapper = this.paramCacheMap.get(paramTypeKey);
		if (paramModelWrapper != null) {
			if (paramModelWrapper.getContainAll()) {
				paramModelWrapper.getParamModelMap().clear();
			} else {
				paramModelWrapper.getParamModelMap().remove(key);
			}
			paramModelWrapper.setContainAll(false); // 设置为不包含全部
		}
	}
	
	
	/**
	 * 获取缓存类Key.
	 * @param orgId 机构号
	 * @param paramClassName 参数类名
	 * @return 缓存类Key
	 */
	private String getCacheTypeKey(String orgId, String paramClassName) {
		return orgId + "@!@" + paramClassName;
	}
	
	
	/**
	 * 参数缓存模型包裹器.
	 * @author zhongyang
	 *
	 */
	@Data
	@SuppressWarnings("serial")
	private class ParamModelWrapper implements Serializable {
		
		// 机构号
		private String orgId;
		
		// 参数类名
		private String paramClass;
		
		// 参数value集合, paramKey , paramValue
		private ConcurrentHashMap<String, ParamModel> paramModelMap;

		// 参数存储服务接口
		private ParamStoreService paramStoreService;

		// 参数是否包含全部.
		private Boolean containAll;

	}

}
