package org.krproject.ocean.vitamins.param.service.impl;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;

import org.krproject.ocean.vitamins.param.constants.ParamConstants;
import org.krproject.ocean.vitamins.param.domain.enums.ParamOperationEnum;
import org.krproject.ocean.vitamins.param.event.ParamChangedEvent;
import org.krproject.ocean.vitamins.param.service.ParamService;
import org.krproject.ocean.vitamins.param.service.inner.ParamAccessService;
import org.krproject.ocean.vitamins.param.service.inner.ParamPubsubService;
import org.krproject.ocean.vitamins.param.service.inner.ParamStoreService;
import org.krproject.ocean.vitamins.param.service.inner.customize.AbstractParamCustomize;
import org.krproject.ocean.vitamins.param.service.inner.model.ParamModel;
import org.krproject.ocean.vitamins.param.service.inner.model.ParamPubsubModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.integration.annotation.ServiceActivator;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import lombok.extern.slf4j.Slf4j;


/**
 * 参数实现.
 * @author zhongyang
 *
 */
@Slf4j
@Service
public class ParamServiceImpl implements ParamService {

	@Autowired
	private ApplicationContext applicationContext;

	@Autowired
	private ParamAccessService paramAccessService;

	@Autowired
	private ParamPubsubService paramPubsubService;

	@Autowired
	private ParamStoreService paramStoreService;

	@Autowired(required = false)
	private List<AbstractParamCustomize<?>> paramCustomizes;
	
	private Map<String, AbstractParamCustomize<?>> paramCustomizeMap = new ConcurrentHashMap<String, AbstractParamCustomize<?>>();
	
	@PostConstruct
	public void init() {
		if (this.paramCustomizes != null) {
			for (AbstractParamCustomize<?> paramCustomize : this.paramCustomizes) {
				log.info("Param Customized:{}", paramCustomize.getParamClass().getCanonicalName());
				this.paramCustomizeMap.put(paramCustomize.getParamClass().getCanonicalName(), paramCustomize);
			}
		}
	}
	
	
	@Override
	public <T> Map<String, T> getParameterMap(String orgId, Class<T> paramClass) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		
		// 获取参数类名
		String paramClassName = paramClass.getCanonicalName();
		
		// 调用参数获取服务获取参数列表
		AbstractParamCustomize<?> paramCustomize = this.paramCustomizeMap.get(paramClassName);
		if (paramCustomize != null) {
			return paramCustomize.getParameterMap(orgId, paramClassName, paramCustomize);
		}
		return this.paramAccessService.getParameterMap(orgId, paramClassName, this.paramStoreService);
	}
	
	@Override
	public <T> Map<String, T> getParameterMap(Class<T> paramClass) {
		return getParameterMap(ParamConstants.DEFAULT_ORG_ID, paramClass);
	}
	
	@Override
	public <T> T getParameter(String orgId, Class<T> paramClass, String key) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		Assert.notNull(key, "指定参数key不能为空");
		
		// 获取参数类名
		String paramClassName = paramClass.getCanonicalName();
		
		// 调用参数获取服务获取参数
		AbstractParamCustomize<?> paramCustomize = this.paramCustomizeMap.get(paramClassName);
		if (paramCustomize != null) {
			return paramCustomize.getParameter(orgId, paramClassName, key, paramCustomize);
		}
		return this.paramAccessService.getParameter(orgId, paramClassName, key, this.paramStoreService);
	}
	
	@Override
	public <T> T getParameter(Class<T> paramClass, String key) {
		return getParameter(ParamConstants.DEFAULT_ORG_ID, paramClass, key);
	}

	
	@Override
	public <T> void addParameter(String orgId, String key, T parameter, String mtnUser) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(parameter, "指定参数parameter不能为空");
		Assert.notNull(key, "指定参数key不能为空");
		
		// 获取参数类名
		String paramClassName = parameter.getClass().getCanonicalName();

		// 调用存储服务添加参数
		ParamModel paramModel = new ParamModel();
		paramModel.setOrgId(orgId);
		paramModel.setParamClass(paramClassName);
		paramModel.setParamKey(key);
		paramModel.setParamObject(parameter);
		paramModel.setMtnUser(mtnUser);
		paramModel.setLastUpdateTime(new Date());
		AbstractParamCustomize<?> paramCustomize = this.paramCustomizeMap.get(paramClassName);
		if (paramCustomize != null) {
			paramCustomize.addParamStore(paramModel);
		} else {
			this.paramStoreService.addParamStore(paramModel);
		}
		
		// 外部广播参数变更模型
		this.paramPubsubService.publish(new ParamPubsubModel(orgId, paramClassName, key, ParamOperationEnum.INSERT));

		// 内部发出参数变更事件
		this.applicationContext.publishEvent(new ParamChangedEvent(this, orgId, paramClassName, key, ParamOperationEnum.INSERT));
	}
	
	@Override
	public <T> void addParameter(String orgId, String key, T parameter) {
		addParameter(orgId, key, parameter, ParamConstants.DEFAULT_MANAGE_USER);
	}
	
	@Override
	public <T> void addParameter(String key, T parameter) {
		addParameter(ParamConstants.DEFAULT_ORG_ID, key, parameter, ParamConstants.DEFAULT_MANAGE_USER);
	}

	
	@Override
	public <T> void updateParameter(String orgId, String key, T parameter, String mtnUser) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(parameter, "指定参数parameter不能为空");
		Assert.notNull(key, "指定参数key不能为空");
		
		// 获取参数类名
		String paramClassName = parameter.getClass().getCanonicalName();
		
		// 调用存储服务更新参数
		ParamModel paramModel = new ParamModel();
		paramModel.setOrgId(orgId);
		paramModel.setParamClass(paramClassName);
		paramModel.setParamKey(key);
		paramModel.setParamObject(parameter);
		paramModel.setMtnUser(mtnUser);
		paramModel.setLastUpdateTime(new Date());
		AbstractParamCustomize<?> paramCustomize = this.paramCustomizeMap.get(paramClassName);
		if (paramCustomize != null) {
			paramCustomize.updateParamStore(paramModel);
		} else {
			this.paramStoreService.updateParamStore(paramModel);
		}

		// 外部广播参数变更模型
		this.paramPubsubService.publish(new ParamPubsubModel(orgId, paramClassName, key, ParamOperationEnum.UPDATE));

		// 内部发出参数变更事件
		this.applicationContext.publishEvent(new ParamChangedEvent(this, orgId, paramClassName, key, ParamOperationEnum.UPDATE));
	}
	
	@Override
	public <T> void updateParameter(String orgId, String key, T parameter) {
		updateParameter(orgId, key, parameter, ParamConstants.DEFAULT_MANAGE_USER);		
	}
	
	@Override
	public <T> void updateParameter(String key, T parameter) {
		updateParameter(ParamConstants.DEFAULT_ORG_ID, key, parameter, ParamConstants.DEFAULT_MANAGE_USER);
	}

	@Override
	public <T> void removeParameter(String orgId, Class<T> paramClass, String key, String mtnUser) {
		Assert.notNull(orgId, "指定参数orgId不能为空");
		Assert.notNull(paramClass, "指定参数paramClass不能为空");
		Assert.notNull(key, "指定参数key不能为空");
		
		// 获取参数类名
		String paramClassName = paramClass.getCanonicalName(); 
		
		// 调用参数存储服务删除参数
		AbstractParamCustomize<?> paramCustomize = this.paramCustomizeMap.get(paramClassName);
		if (paramCustomize != null) {
			paramCustomize.removeParamStore(orgId, paramClassName, key, mtnUser);
		} else {
			this.paramStoreService.removeParamStore(orgId, paramClassName, key, mtnUser);
		}

		// 外部广播参数变更模型
		this.paramPubsubService.publish(new ParamPubsubModel(orgId, paramClassName, key, ParamOperationEnum.DELETE));

		// 内部发出参数变更事件
		this.applicationContext.publishEvent(new ParamChangedEvent(this, orgId, paramClassName, key, ParamOperationEnum.DELETE));
	}
	
	@Override
	public <T> void removeParameter(String orgId, Class<T> paramClazz, String key) {
		removeParameter(orgId, paramClazz, key, ParamConstants.DEFAULT_MANAGE_USER);
	}
	
	@Override
	public <T> void removeParameter(Class<T> paramClazz, String key) {
		removeParameter(ParamConstants.DEFAULT_ORG_ID, paramClazz, key, ParamConstants.DEFAULT_MANAGE_USER);
	}
	
	
	// 参数变更订阅处理
	@ServiceActivator(inputChannel = ParamConstants.PARAM_PUBSUB_CHANNEL_NAME)
	public void subscribeEndpoint(ParamPubsubModel paramPubsubModel) {
		Assert.notNull(paramPubsubModel, "广播订阅模型不能为空");
		Assert.notNull(paramPubsubModel.getOrgId(), "广播订阅模型orgId不能为空");
		Assert.notNull(paramPubsubModel.getParamClass(), "广播订阅模型paramClass不能为空");
		
		// 获取参数类名
		String paramClassName = paramPubsubModel.getParamClass(); 
		
		// 调用参数广播订阅服务订阅参数变更
		AbstractParamCustomize<?> paramCustomize = this.paramCustomizeMap.get(paramClassName);
		if (paramCustomize != null) {
			paramCustomize.subscribe(paramPubsubModel);
		} else {
			this.paramPubsubService.subscribe(paramPubsubModel);
		}
	}

}
