package org.krproject.ocean.vitamins.online.api;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.domain.Sort.Order;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * 分页请求对象，支持序列化.
 * 
 * @author zhongyang
 *
 */
@SuppressWarnings("serial")
public class OnlinePageRequest implements Pageable {
	
	private int page;
	
	private int size;
	
	@JsonProperty("sort")
	private List<PageOrder> pageOrders;
	
	public int getPage() {
		return this.page;
	}

	public void setPage(int page) {
		this.page = page;
	}

	public int getSize() {
		return this.size;
	}

	public void setSize(int size) {
		this.size = size;
	}

	public List<PageOrder> getPageOrders() {
		return this.pageOrders;
	}

	public void setPageOrders(List<PageOrder> pageOrders) {
		this.pageOrders = pageOrders;
	}

	public OnlinePageRequest() {
		this.pageOrders = new ArrayList<>();
	}

	public OnlinePageRequest(int page, int size) {
		this.page = page;
		this.size = size;
		this.pageOrders = new ArrayList<>();
	}
	
	public OnlinePageRequest(Pageable pageable) {
		this.page = pageable.getPageNumber();
		this.size = pageable.getPageSize();
		if (pageable.getSort() != null && pageable.getSort().iterator() != null) {
			Iterator<Order> iterator = pageable.getSort().iterator();
			List<PageOrder> pageOrders = new ArrayList<>();
			while (iterator.hasNext()) {
				Order order = iterator.next();
				PageOrder pageOrder = new PageOrder();
				pageOrder.setDirection(order.getDirection());
				pageOrder.setProperty(order.getProperty());
				pageOrders.add(pageOrder);
			}
			this.pageOrders = pageOrders;
		} else {
			this.pageOrders = new ArrayList<>();
		}
	}
	
	public OnlinePageRequest(int page, int size, List<PageOrder> pageOrders) {
		this.page = page;
		this.size = size;
		this.pageOrders = pageOrders;
	}
	
	@JsonIgnore
	@Override
	public boolean isPaged() {
		return Pageable.super.isPaged();
	}
	
	@JsonIgnore
	@Override
	public boolean isUnpaged() {
		return Pageable.super.isUnpaged();
	}

	@JsonIgnore
	@Override
	public int getPageNumber() {
		return this.page;
	}

	@JsonIgnore
	@Override
	public int getPageSize() {
		return this.size;
	}

	@JsonIgnore
	@Override
	public long getOffset() {
		return (this.page * this.size);
	}

	@JsonIgnore
	@Override
	public Sort getSort() {
		Sort sort = null;
		if (this.pageOrders != null && !this.pageOrders.isEmpty()) {
			List<Order> orders = new ArrayList<>();
			for (PageOrder pageOrder : this.pageOrders) {
				Order order = new Order(pageOrder.getDirection(), pageOrder.getProperty());
				orders.add(order);
			}
			sort = Sort.by(orders);
		} else {
			sort = Sort.unsorted();
		}
		return sort;
	}

	@Override
	public Pageable withPage(int pageNumber) {
		return new OnlinePageRequest(pageNumber, getPageSize(), this.pageOrders);
	}
	
	@Override
	public Pageable next() {
		return new OnlinePageRequest(getPageNumber() + 1, getPageSize(), this.pageOrders);
	}
	
	public Pageable previous() {
		return getPageNumber() == 0 ? this : new OnlinePageRequest(getPageNumber() - 1, getPageSize(), this.pageOrders);
	}

	@Override
	public Pageable previousOrFirst() {
		return ((hasPrevious()) ? previous() : first());
	}

	@Override
	public Pageable first() {
		return new OnlinePageRequest(0, getPageSize(), this.pageOrders);
	}

	@Override
	public boolean hasPrevious() {
		return (this.page > 0);
	}
	

	

	/**
	 * PageOrder.
	 * @author zhongyang
	 *
	 */
	public static class PageOrder implements Serializable {
		
		private Direction direction;
		
		private String property;
		
		public Direction getDirection() {
			return this.direction;
		}

		public void setDirection(Direction direction) {
			this.direction = direction;
		}

		public String getProperty() {
			return this.property;
		}

		public void setProperty(String property) {
			this.property = property;
		}

	}
}
