package org.krproject.ocean.vitamins.domain.config;

import java.sql.SQLException;

import org.h2.tools.Server;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import lombok.extern.slf4j.Slf4j;

/**
 * H2服务配置文件.
 * 
 * @author Tiger
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(DomainH2ServerProperties.class)
public class DomainH2ServerConfig implements DisposableBean  {
	
	@Autowired
	private DataSourceProperties dataSourceProperties;
	
	@Autowired
	private DomainH2ServerProperties h2ServerProperties;

	private Server tcpServer = null;
	
	private Server webServer = null;
	
	/**
	 * TCP connection to connect with SQL clients to the embedded h2 database.
	 *
	 * Connect to "jdbc:h2:tcp://localhost:9092/mem:testdb", username "sa", password empty.
	 * @return Server
	 * @throws SQLException SQLException
	 */
	@Bean
	@ConditionalOnMissingBean(name = "krprojectOceanH2TcpServer")
	@ConditionalOnProperty(prefix = "krproject.ocean.domain.h2", name = "tcp-enabled", matchIfMissing = false)
	public Server krprojectOceanH2TcpServer() throws SQLException {
		this.tcpServer = Server.createTcpServer("-tcp", "-tcpAllowOthers", "-tcpPort", this.h2ServerProperties.getTcpPort()).start();
		log.info("h2 tcp started with url: {}", this.dataSourceProperties.determineUrl());
		return this.tcpServer;
	}

	/**
	 * Web console for the embedded h2 database.
	 * Go to http://localhost:8082 and connect to the database "jdbc:h2:mem:testdb", username "sa", password empty.
	 * @return Server
	 * @throws SQLException SQLException
	 */
	@Bean
	@ConditionalOnMissingBean(name = "krprojectOceanH2WebServer")
	@ConditionalOnProperty(prefix = "krproject.ocean.domain.h2", name = "web-enabled", matchIfMissing = false)
	public Server krprojectOceanH2WebServer() throws SQLException {
		this.webServer = Server.createWebServer("-web", "-webAllowOthers", "-webPort", this.h2ServerProperties.getWebPort()).start();
		log.info("h2 web started with url: {}", this.dataSourceProperties.determineUrl());
		return this.webServer;
	}

	@Override
	public void destroy() throws Exception {
		if (this.tcpServer != null) {
			this.tcpServer.stop();
			this.tcpServer.shutdown();
			log.info("h2 tcp stopped");
		}
		
		if (this.webServer != null) {
			this.webServer.stop();
			this.webServer.shutdown();
			log.info("h2 web stopped");
		}
	}

}
