package org.krproject.ocean.vitamins.admin.service.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.krproject.ocean.vitamins.admin.constants.AdminConstants;
import org.krproject.ocean.vitamins.admin.service.AdminSecurityService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.oauth2.provider.ClientDetails;
import org.springframework.security.oauth2.provider.ClientRegistrationException;
import org.springframework.security.oauth2.provider.client.BaseClientDetails;

import lombok.extern.slf4j.Slf4j;


/**
 * 后管安全服务的假实现.

 * 注意：此为假的实现，真实环境需要替换.
 * 
 * @author Tiger
 */
@Slf4j
public class FakeAdminSecurityServiceImpl implements AdminSecurityService {

	@Autowired
	@Qualifier(AdminConstants.CLIENT_PASSWORD_ENCODER)
	private PasswordEncoder clientPasswordEncoder;

	@Autowired
	@Qualifier(AdminConstants.USER_PASSWORD_ENCODER)
	private PasswordEncoder userPasswordEncoder;

	private static final Map<String, FakeRole> FAKE_ROLES = new HashMap<String, FakeRole>();
	private static final Map<String, FakePriv> FAKE_PRIVS = new HashMap<String, FakePriv>();
	
	@Override
	public ClientDetails loadClientByClientId(String clientId) throws ClientRegistrationException {
		
		log.debug("loadClientByClientId:{}", clientId);
		
		BaseClientDetails details = new BaseClientDetails();
		details.setClientId(clientId);
		details.setClientSecret(this.clientPasswordEncoder.encode(clientId));
		details.setResourceIds(Arrays.asList(AdminConstants.OAUTH2_RESOURCE_ID));
		details.setScope(Arrays.asList("read", "write", "trust"));
		details.setAuthorizedGrantTypes(Arrays.asList("client_credentials", "password", "refresh_token"));
		details.setAccessTokenValiditySeconds(10000);
		details.setRefreshTokenValiditySeconds(60000);
		return details;
	}

	@Override
	public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
		
		log.debug("loadUserByUsername:{}", username);

		// 用户拥有所有权限
		List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();
		for (FakePriv priv : FAKE_PRIVS.values()) {
			authorities.add(new SimpleGrantedAuthority(priv.name));
		}
		
		return new User(username, this.userPasswordEncoder.encode(username), authorities);
	}

	@Override
	public void createPriv(String privName, String privDesc) {
		FakePriv priv = new FakePriv();
		priv.name = privName;
		priv.desc = privDesc;
		FAKE_PRIVS.put(priv.name, priv);
	}

	@Override
	public void createRole(String roleName, String roleDesc, Collection<String> prives) {
		FakeRole role = new FakeRole();
		role.name = roleName;
		role.desc = roleDesc;
		role.privs = prives;
		FAKE_ROLES.put(role.name, role);
	}
	
	/**
	 * 权限定义.
	 * @author Tiger
	 *
	 */
	public class FakePriv {
		String name;
		String desc;
	}
	
	/**
	 * 角色定义.
	 * @author Tiger
	 *
	 */
	public class FakeRole {
		String name;
		String desc;
		Collection<String> privs;
	}

}
