package org.krproject.ocean.vitamins.admin.service.impl;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Function;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.krproject.ocean.vitamins.admin.exception.AdminException;
import org.krproject.ocean.vitamins.admin.model.AdminRespCode;
import org.krproject.ocean.vitamins.admin.model.param.ParamAuditVo;
import org.krproject.ocean.vitamins.admin.model.param.ParamVo;
import org.krproject.ocean.vitamins.admin.service.AdminParamService;
import org.krproject.ocean.vitamins.param.domain.ParamAuditEntity;
import org.krproject.ocean.vitamins.param.domain.ParamAuditRepository;
import org.krproject.ocean.vitamins.param.domain.enums.ParamOperationEnum;
import org.krproject.ocean.vitamins.param.exception.ErrorCode;
import org.krproject.ocean.vitamins.param.exception.ParamException;
import org.krproject.ocean.vitamins.param.service.ParamService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

import lombok.extern.slf4j.Slf4j;


/**
 * 参数服务接口实现类.
 * @author zhongyang
 *
 */
@Slf4j
@Service
public class AdminParamServiceImpl implements AdminParamService {

	private static final ObjectMapper objectMapper = new ObjectMapper();
	
	static {
		objectMapper.configure(SerializationFeature.WRAP_ROOT_VALUE, true);
		objectMapper.configure(DeserializationFeature.UNWRAP_ROOT_VALUE, true);
		objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
	}
	
	@Autowired
	private ParamService paramService;
	
	@Autowired
	private ParamAuditRepository paramAuditRepository;
	
	@Override
	public List<ParamVo> getParameterList(String orgId, String paramClassName) {
		
		Class<?> paramClass = null;
		try {
			paramClass = Class.forName(paramClassName);
		} catch (ClassNotFoundException e) {
			log.error("get class if name {0} failed:{1}!", paramClassName, e);
			throw new AdminException(AdminRespCode.NOT_FOUND, MessageFormat.format("get class for name:{0} failed!", paramClassName));
		}
		
		List<ParamVo> paramList = new ArrayList<>();
		try {
			Map<String, ?> paramMap = this.paramService.getParameterMap(orgId, paramClass);
			if (paramMap != null && !paramMap.isEmpty()) {
				for (Entry<String, ?> paramSet : paramMap.entrySet()) {
					ParamVo param = paramConveter(orgId, paramClassName, paramSet.getKey(), paramSet.getValue(), null);
					paramList.add(param);
				}
				return paramList;
			}
		} catch (ParamException e) {
			log.error("getParameterMap orgId:{0} paramClass:{1} failed:{2}!", orgId, paramClass, e);
			throw new AdminException(AdminRespCode.NOT_FOUND, MessageFormat.format("getParameterMap orgId:{0} paramClass:{1} failed!", orgId, paramClass));
		}
		return paramList;
	}

	@Override
	public ParamVo getParameter(String orgId, String paramClassName, String key) {
		
		Class<?> paramClass = null;
		try {
			paramClass = Class.forName(paramClassName);
		} catch (ClassNotFoundException e) {
			log.error("get class if name {0} failed:{1}!", paramClassName, e);
			throw new AdminException(AdminRespCode.NOT_FOUND, MessageFormat.format("get class for name:{0} failed!", paramClassName));
		}
		
		ParamVo param = null;
		try {
			Object paramObject = this.paramService.getParameter(orgId, paramClass, key);
			if (paramObject != null) {
				param = paramConveter(orgId, paramClassName, key, paramObject, null);
				return param;
			}
		} catch (ParamException e) {
			log.error("getParameterMap orgId:{0} paramClass:{1} key:{2} failed:{3}!", orgId, paramClass, key, e);
			throw new AdminException(AdminRespCode.NOT_FOUND, MessageFormat.format("getParameterMap orgId:{0} paramClass:{1} key:{2} failed!", orgId, paramClass, key));
		}
		return param;
	}

	@Override
	public void addParameter(ParamVo paramVo) {
		
		// 参数校验
		if (!StringUtils.hasLength(paramVo.getOrgId())) {
			throw new AdminException(AdminRespCode.FAILURE, "param orgId is empty!");
		}
		if (!StringUtils.hasLength(paramVo.getParamClassName())) {
			throw new AdminException(AdminRespCode.FAILURE, "param paramClassName is empty!");
		}
		if (!StringUtils.hasLength(paramVo.getParamKey())) {
			throw new AdminException(AdminRespCode.FAILURE, "param paramKey is empty!");
		}
		if (!StringUtils.hasLength(paramVo.getParamObject())) {
			throw new AdminException(AdminRespCode.FAILURE, "param paramObject is empty!");
		}
		try {
			Object newObject = unmarshal(paramVo.getParamObject(), paramVo.getParamClassName());
			this.paramService.addParameter(paramVo.getOrgId(), paramVo.getParamKey(), newObject, paramVo.getMtnUser());
		} catch (ParamException e) {
			log.error("addParameter orgId:{0} newObject:{1} key:{2} failed:{3}!", 
					paramVo.getOrgId(), paramVo.getParamObject(), paramVo.getParamKey(), e);
			throw new AdminException(AdminRespCode.NOT_FOUND, 
					MessageFormat.format("addParameter orgId:{0} newObject:{1} key:{2} failed!", 
							paramVo.getOrgId(), paramVo.getParamObject(), paramVo.getParamKey()));
		} catch (Exception e) {
			log.error("addParameter orgId:{0} newObject:{1} key:{2} failed:{3}!", 
					paramVo.getOrgId(), paramVo.getParamObject(), paramVo.getParamKey(), e);
			throw new AdminException(AdminRespCode.FAILURE, e.getLocalizedMessage());
		}
		
	}

	@Override
	public void updateParameter(ParamVo paramVo) {
		
		// 参数校验
		if (!StringUtils.hasLength(paramVo.getOrgId())) {
			throw new AdminException(AdminRespCode.FAILURE, "param orgId is empty!");
		}
		if (!StringUtils.hasLength(paramVo.getParamClassName())) {
			throw new AdminException(AdminRespCode.FAILURE, "param paramClassName is empty!");
		}
		if (!StringUtils.hasLength(paramVo.getParamKey())) {
			throw new AdminException(AdminRespCode.FAILURE, "param paramKey is empty!");
		}
		if (!StringUtils.hasLength(paramVo.getParamObject())) {
			throw new AdminException(AdminRespCode.FAILURE, "param paramObject is empty!");
		}
		
		try {
			Object newObject = unmarshal(paramVo.getParamObject(), paramVo.getParamClassName());
			this.paramService.updateParameter(paramVo.getOrgId(), paramVo.getParamKey(), newObject, paramVo.getMtnUser());
		} catch (ParamException e) {
			log.error("updateParameter orgId:{0} newObject:{1} key:{2} failed:{3}!", 
					paramVo.getOrgId(), paramVo.getParamObject(), paramVo.getParamKey(), e);
			throw new AdminException(AdminRespCode.NOT_FOUND, 
					MessageFormat.format("updateParameter orgId:{0} newObject:{1} key:{2} failed!", 
							paramVo.getOrgId(), paramVo.getParamObject(), paramVo.getParamKey()));
		} catch (Exception e) {
			log.error("updateParameter orgId:{0} newObject:{1} key:{2} failed:{3}!", 
					paramVo.getOrgId(), paramVo.getParamObject(), paramVo.getParamKey(), e);
			throw new AdminException(AdminRespCode.FAILURE, e.getLocalizedMessage());
		}
	}

	@Override
	public void removeParameter(String orgId, String paramClassName, String key, String mtnUser) {
		
		Class<?> paramClass = null;
		try {
			paramClass = Class.forName(paramClassName);
		} catch (ClassNotFoundException e) {
			log.error("get class if name {0} failed:{1}!", paramClassName, e);
			throw new AdminException(AdminRespCode.NOT_FOUND, MessageFormat.format("get class for name:{0} failed!", paramClassName));
		}
		
		try {
			this.paramService.removeParameter(orgId, paramClass, key, mtnUser);
		} catch (ParamException e) {
			log.error("getParameterMap orgId:{0} paramClass:{1} key:{2} failed:{3}!", orgId, paramClass, key, e);
			throw new AdminException(AdminRespCode.NOT_FOUND, MessageFormat.format("getParameterMap orgId:{0} paramClass:{1} key:{2} failed!", orgId, paramClass, key));
		}
	}

	
	@Override
	public Page<ParamAuditVo> pageParamAudit(String orgId, String paramClass, String paramKey,
			ParamOperationEnum paramOperation, Pageable pageable) {
		//构造动态查询条件
		@SuppressWarnings("serial")
		Specification<ParamAuditEntity> spec = new Specification<ParamAuditEntity>() {

			@Override
			public Predicate toPredicate(Root<ParamAuditEntity> root, CriteriaQuery<?> query,
					CriteriaBuilder cb) {
				List<Predicate> predicates = new ArrayList<Predicate>();
				if (StringUtils.hasLength(orgId)) {
					predicates.add(cb.equal(root.get("orgId"), orgId));
				}
				if (StringUtils.hasLength(paramClass)) {
					predicates.add(cb.equal(root.get("paramClass"), paramClass));
				}
				if (StringUtils.hasLength(paramKey)) {
					predicates.add(cb.equal(root.get("paramKey"), paramKey));
				}
				if (paramOperation != null) {
					predicates.add(cb.equal(root.get("paramOperation"), paramOperation));
				}
				return query.where(predicates.toArray(new Predicate[predicates.size()])).getRestriction();
			}
			
		};
		return this.paramAuditRepository.findAll(spec, pageable).map(new ParamAuditConverter());
	}
	
	private String marshal(Object paramObject) {
		String str = null;
		try {
			str = objectMapper.writeValueAsString(paramObject);
		} catch (Exception e) {
			log.error("marshal object:{} failed!", paramObject);
			throw new ParamException(ErrorCode.PARAM_PARSE_ERROR, "marshal paramObject failed!");
		}
		return str;
	}


	private Object unmarshal(String paramObjectStr, String paramObjectClz) {
		Object obj = null;
		try {
			Class<?> clazz = Class.forName(paramObjectClz);
			obj = objectMapper.readValue(paramObjectStr, clazz);
		} catch (Exception e) {
			log.error("unmarshal message:{} class:{} failed!", paramObjectStr, paramObjectClz);
			throw new ParamException(ErrorCode.PARAM_PARSE_ERROR, "unmarshal paramObject failed!");
		}
		return obj;
	}
	
	/**
	 * 填充param.
	 * @param orgId orgId
	 * @param paramClassName paramClassName
	 * @param paramKey paramKey
	 * @param paramObject paramObject
	 * @param mtnUser mtnUser
	 * @return Param Param
	 */
	private ParamVo paramConveter(String orgId, String paramClassName, String paramKey, Object paramObject, String mtnUser) {
		ParamVo param = new ParamVo();
		param.setOrgId(orgId);
		param.setParamClassName(paramClassName);
		param.setParamKey(paramKey);
		param.setParamObject(marshal(paramObject));
		param.setMtnUser(mtnUser);
		return param;
	}
	
	/**
	 * ParamAuditConverter.
	 * @author zhongyang
	 *
	 */
	private class ParamAuditConverter implements Function<ParamAuditEntity, ParamAuditVo> {

		@Override
		public ParamAuditVo apply(ParamAuditEntity audit) {
			if (audit != null) {
				ParamAuditVo auditVo = new ParamAuditVo();
				auditVo.setAuditId(audit.getAuditId());
				auditVo.setOrgId(audit.getOrgId());
				auditVo.setParamClass(audit.getParamClass());
				auditVo.setParamKey(audit.getParamKey());
				auditVo.setParamOperation(audit.getParamOperation());
				auditVo.setOldObject(audit.getOldObject());
				auditVo.setNewObject(audit.getNewObject());
				auditVo.setMtnUser(audit.getMtnUser());
				auditVo.setMtnMemo(audit.getMtnMemo());
				auditVo.setCreateTime(audit.getCreateTime());
				return auditVo;
			}
			return null;
		}
		
	}

}
