package org.krproject.ocean.vitamins.admin.controller.param;

import java.security.Principal;
import java.util.ArrayList;
import java.util.List;

import org.krproject.ocean.vitamins.admin.annotation.SecurityPriv;
import org.krproject.ocean.vitamins.admin.annotation.SecurityRole;
import org.krproject.ocean.vitamins.admin.model.AdminResponse;
import org.krproject.ocean.vitamins.admin.model.param.ParamAuditVo;
import org.krproject.ocean.vitamins.admin.model.param.ParamVo;
import org.krproject.ocean.vitamins.admin.service.AdminParamService;
import org.krproject.ocean.vitamins.param.domain.enums.ParamOperationEnum;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.web.PageableDefault;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;


/**
 * 参数处理器.
 * 
 * @author Tiger
 */
@Slf4j
@ConditionalOnProperty(prefix = "krproject.ocean.admin", name = "param-enabled", matchIfMissing = true)
@Tag(name = "/param")
@RequestMapping(value = "/param", produces = {MediaType.APPLICATION_JSON_VALUE})
@RestController
@SecurityRole(name = "ROLE_KR_PARAM", desc = "通用参数_角色", prives = {
		@SecurityPriv(name = "PRIV_KR_PARAM_READ", desc = "通用参数_读权限"), 
		@SecurityPriv(name = "PRIV_KR_PARAM_WRITE", desc = "通用参数_写权限")}
)
public class AdminParamController {

	@Autowired
	private AdminParamService adminParamService;
	
	
	@Secured("PRIV_KR_PARAM_READ")
	@RequestMapping(value = "/list", method = RequestMethod.GET, name = "获取一类参数")
	public ResponseEntity<AdminResponse> getParams(
			@RequestParam(required = true) String orgId,
			@RequestParam(required = true) String paramClassName,
			@RequestParam(required = false) String paramKey) {
		log.debug("list params of orgId:{} paramClassName:{}", orgId, paramClassName);
		
		List<ParamVo> paramList = new ArrayList<>();
		if (StringUtils.hasLength(paramKey)) {
			ParamVo param = this.adminParamService.getParameter(orgId, paramClassName, paramKey);
			if (param != null) {
				paramList.add(param);
			}
		} else {
			List<ParamVo> params = this.adminParamService.getParameterList(orgId, paramClassName);
			if (params != null && params.size() > 0) {
				paramList = params;
			}
		}
		Page<ParamVo> paramPage = new PageImpl<ParamVo>(paramList);
		
		return ResponseEntity.ok(new AdminResponse(paramPage));
	}
	
	@Secured("PRIV_KR_PARAM_READ")
	@RequestMapping(value = "", method = RequestMethod.GET, name = "获取参数")
	public ResponseEntity<AdminResponse> getParam(
			@RequestParam(required = true) String orgId,
			@RequestParam(required = true) String paramClassName,
			@RequestParam(required = true) String paramKey) {
		log.debug("get param of orgId:{} paramClassName:{} paramKey:{}", orgId, paramClassName, paramKey);
		
		ParamVo param = this.adminParamService.getParameter(orgId, paramClassName, paramKey);
		
		return ResponseEntity.ok(new AdminResponse(param));
	}
	
	@Secured("PRIV_KR_PARAM_WRITE")
	@RequestMapping(value = "", method = RequestMethod.POST, name = "新增参数")
	public ResponseEntity<AdminResponse> createParam(
			@RequestBody(required = true) @Validated ParamVo param, Principal principal) {
		log.debug("create param:{}", param);
		
		String mtnUser = (principal == null) ? null : principal.getName();
		param.setMtnUser(mtnUser);
		this.adminParamService.addParameter(param);
		
		return ResponseEntity.ok(new AdminResponse());
	}
	
	@Secured("PRIV_KR_PARAM_WRITE")
	@RequestMapping(value = "", method = RequestMethod.PATCH, name = "更新参数")
	public ResponseEntity<AdminResponse> updateParam(
			@RequestBody(required = true) @Validated ParamVo param, Principal principal) {
		log.debug("update param:{}", param);
		
		String mtnUser = (principal == null) ? null : principal.getName();
		param.setMtnUser(mtnUser);
		this.adminParamService.updateParameter(param);
		
		return ResponseEntity.ok(new AdminResponse());
	}
	
	@Secured("PRIV_KR_PARAM_WRITE")
	@RequestMapping(value = "", method = RequestMethod.DELETE, name = "删除参数")
	public ResponseEntity<AdminResponse> deleteParam(
			@RequestParam(required = true) String orgId,
			@RequestParam(required = true) String paramClassName,
			@RequestParam(required = true) String paramKey,
			Principal principal) {
		log.debug("delete param of orgId:{} paramClassName:{} paramKey:{}", orgId, paramClassName, paramKey);
		
		String mtnUser = (principal == null) ? null : principal.getName();
		this.adminParamService.removeParameter(orgId, paramClassName, paramKey, mtnUser);
		
		return ResponseEntity.ok(new AdminResponse());
	}
	
	@Secured("PRIV_KR_PARAM_READ")
	@RequestMapping(value = "/audits", method = RequestMethod.GET, name = "查询参数审计列表")
	public ResponseEntity<AdminResponse> listParamAudit(
			@RequestParam(required = false) String orgId,
			@RequestParam(required = false) String paramClass,
			@RequestParam(required = false) String paramKey,
			@RequestParam(required = false) ParamOperationEnum paramOperation,
			@PageableDefault(sort = {"auditId"}, direction = Direction.DESC) Pageable pageable) {
		log.debug("list Param Audits:{}", pageable);
		
		Page<ParamAuditVo> page = this.adminParamService.pageParamAudit(orgId, paramClass, paramKey, paramOperation, pageable);
		
		return ResponseEntity.ok(new AdminResponse(page));
	}

}
