package org.krproject.ocean.vitamins.admin.controller;

import org.krproject.ocean.vitamins.admin.exception.AdminException;
import org.krproject.ocean.vitamins.admin.model.AdminRespCode;
import org.krproject.ocean.vitamins.admin.model.AdminResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import lombok.extern.slf4j.Slf4j;


/**
 * 全局统一异常处理器.
 * Central class for behavior common to all REST controllers.
 * <p>
 * Note that any controller-specific exception handler is resolved first. 
 * So for example, having a onException(Exception e) resolver at a controller level 
 * will prevent the one from this class to be triggered.
 * </p>
 *
 * @author Tiger
 */
@Slf4j
@RestControllerAdvice
public class AdminControllerGlobalAdvice {

	/**
	 * Handles the case where client does not login.
	 *
	 * @param e exception to be handled
	 *
	 * @return RestyResponse see {@link AdminResponse}
	 */
	@ExceptionHandler(AuthenticationException.class)
	public ResponseEntity<AdminResponse> onAuthenticationException(AuthenticationException e) {
		log.debug("认证失败", e);
		return ResponseEntity.status(AdminRespCode.UNAUTHORIZED.getHttpStatus())
				.body(new AdminResponse(AdminRespCode.UNAUTHORIZED, e.getMessage()));
	}
	
	
	/**
	 * Handles the case where client has no permissions.
	 *
	 * @param e exception to be handled
	 *
	 * @return RestyResponse see {@link AdminResponse}
	 */
	@ExceptionHandler(AccessDeniedException.class)
	public ResponseEntity<AdminResponse> onAccessDeniedException(AccessDeniedException e) {
		log.debug("鉴权失败", e);
		return ResponseEntity.status(AdminRespCode.FORBIDDEN.getHttpStatus())
				.body(new AdminResponse(AdminRespCode.FORBIDDEN, e.getMessage()));
	}
	
	
	/**
	 * Handles the case where client submitted an ill valued request (missing parameter).
	 *
	 * @param e exception to be handled
	 *
	 * @return RestyResponse see {@link AdminResponse}
	 */
	@ExceptionHandler(MissingServletRequestParameterException.class)
	public ResponseEntity<AdminResponse> onMissingServletRequestParameterException(MissingServletRequestParameterException e) {
		log.debug("请求不合法", e);
		return ResponseEntity.status(AdminRespCode.BAD_REQUEST.getHttpStatus())
				.body(new AdminResponse(AdminRespCode.BAD_REQUEST, e.getMessage()));
	}
	
	
	/**
	 * Handles the Exception to be thrown when validation on an argument annotated with @Valid fails.
	 *
	 * @param e MethodArgumentNotValidException
	 *
	 * @return RestyResponse see {@link AdminResponse}
	 */
	@ExceptionHandler(MethodArgumentNotValidException.class)
	public ResponseEntity<AdminResponse> onMethodArgumentNotValidException(MethodArgumentNotValidException e) {
		log.debug("请求不满足校验规则", e);
		return ResponseEntity.status(AdminRespCode.BAD_REQUEST.getHttpStatus())
				.body(new AdminResponse(AdminRespCode.BAD_REQUEST,  e.getBindingResult().getFieldError().getDefaultMessage()));
	}
	
	/**
	 * Handles the rest exception case.
	 *
	 * @param e the rest exception to be handled
	 *
	 * @return RestResponse see {@link AdminResponse}
	 */
	@ExceptionHandler(AdminException.class)
	public ResponseEntity<AdminResponse> onAdminException(AdminException e) {
		log.debug("自定义Admin异常", e);
		return ResponseEntity.status(e.getRespCode().getHttpStatus())
				.body(new AdminResponse(e.getRespCode(), e.getMessage()));
	}

	/**
	 * Handles the general error case. Report server-side error.
	 *
	 * @param e the exception to be handled
	 *
	 * @return RestyResponse see {@link AdminResponse}
	 */
	@ExceptionHandler(Exception.class)
	public ResponseEntity<AdminResponse> onException(Exception e) {
		log.error("系统内部错误", e);
		return ResponseEntity.status(AdminRespCode.SYSTEM_ERROR.getHttpStatus())
				.body(new AdminResponse(AdminRespCode.SYSTEM_ERROR, e.getMessage()));
	}

}
