package org.krproject.ocean.vitamins.admin.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.PostConstruct;

import org.springdoc.core.GroupedOpenApi;
import org.springdoc.core.SpringDocConfigProperties;
import org.springdoc.core.SpringDocUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.security.OAuthFlow;
import io.swagger.v3.oas.models.security.OAuthFlows;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;


/**
 * Admin Swagger 配置.
 * 
 * Using SpringDoc: https://springdoc.org/
 * 
 * @author Tiger
 */
@ConditionalOnProperty(prefix = "krproject.ocean.admin", name = "swagger-enabled", matchIfMissing = false)
@Configuration
public class AdminSwaggerConfig  {

	@Autowired
	private ServerProperties serverProperties;
	
	@Autowired
	private SpringDocConfigProperties springDocConfigProperties;
	
	@Autowired
	private ConfigurableApplicationContext context;
	
	@PostConstruct
	public void postConstruct() {
		// Creating Docket Dynamically per Rest Controller
		Map<String, Object> controllers = this.context.getBeansWithAnnotation(Tag.class);
		for (Object obj : controllers.values()) {
			Tag rm = ClassUtils.getUserClass(obj).getAnnotation(Tag.class);
			if (rm != null && StringUtils.hasLength(rm.name())) {
				String groupName = rm.name();
				GroupedOpenApi  docket = GroupedOpenApi.builder()
						.group(groupName)
						.pathsToMatch(groupName + "/**")
						.build();
				this.context.getBeanFactory().registerSingleton(groupName + "_docket_api", docket);
			}
		}
	}
	
	@Bean
	public OpenAPI openApi() {
		// Ignore OAuth Path, See: https://stackoverflow.com/questions/65361769/enable-oauth-token-endpoint-springdoc-openapi-ui 
		if (this.springDocConfigProperties.getPathsToExclude() == null) {
			this.springDocConfigProperties.setPathsToExclude(new ArrayList<String>());
		}
		this.springDocConfigProperties.getPathsToExclude().add("/oauth/*");
		
		// Support Spring Data Pageable, See: https://springdoc.org/#spring-data-rest-support
		SpringDocUtils.getConfig().replaceWithClass(
				org.springframework.data.domain.Pageable.class,
				org.springdoc.core.converters.models.Pageable.class);

		// Using OAuth2 Password Flow, See: https://github.com/springdoc/springdoc-openapi/issues/740
		String tokenUrl = "/oauth/token"; // spring security oauth2 default token path
		if (this.serverProperties.getServlet().getContextPath() != null) {
			tokenUrl = this.serverProperties.getServlet().getContextPath() + tokenUrl;
		}
		
		String securitySchemaName = "OAuth2 Password Flow";
		Map<String, SecurityScheme> securitySchemes = new HashMap<String, SecurityScheme>();
		SecurityScheme passwordFlowScheme = new SecurityScheme()
				.type(SecurityScheme.Type.OAUTH2)
				.flows(new OAuthFlows().password(new OAuthFlow().tokenUrl(tokenUrl)));
		securitySchemes.put(securitySchemaName, passwordFlowScheme);
		
		// Default Docket to show all.
		return new OpenAPI()
				.components(new Components().securitySchemes(securitySchemes))
				.addSecurityItem(new SecurityRequirement().addList(securitySchemaName))
				.info(new Info().title("Admin API Documentation").description(
						"<p>后管接口文档，采用 Oauth2 Password 认证模式。</p>" + 
						"<p><b>注意：本功能页面应仅用于开发测试环境，生产环境应关闭本功能页面！</b></p>"));
	}
	
}
