package org.krproject.ocean.vitamins.admin.excel;

import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.net.URLEncoder;
import java.util.List;

import javax.servlet.http.HttpServletResponse;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;

/**
 * Excel工具类.
 * 
 * @author Tiger
 */
public class ExcelUtils {

	/**
	 * 写入HTTP输出流.
	 * 
	 * @param <T> 数据类型
	 * @param response HTTP响应
	 * @param dataList 数据列表
	 * @param fileName 文件名
	 * @param sheetName 表单名
	 * @throws IOException 异常
	 */
	public static <T> void writeHttpResponse(HttpServletResponse response, List<T> dataList, String fileName, String sheetName) throws IOException {
		HSSFWorkbook wb = createHSSFWorkbook(dataList, sheetName);
		response.setContentType("application/vnd.ms-excel;charset=UTF-8");
		response.setHeader("Content-Disposition", String.format("attachment;filename=%s", URLEncoder.encode(fileName, "UTF-8")));
		OutputStream os = response.getOutputStream(); 
		wb.write(os);
		os.flush();
		os.close();
	}
	
	/**
	 * 创建工作簿.
	 * @param <T> 数据类型
	 * @param dataList 数据列表
	 * @param sheetName 表单名
	 * @return 工作簿
	 */
	public static <T> HSSFWorkbook createHSSFWorkbook(List<T> dataList, String sheetName) {
		
		if (dataList == null || dataList.isEmpty()) {
			return null;
		}
		Class<?> clazz = dataList.get(0).getClass();
		Field[] fields = clazz.getDeclaredFields();
		
		// 声明一个工作薄
		HSSFWorkbook wb = new HSSFWorkbook();
		
		// 声明一个单子并命名
		ExcelClass excelClass = clazz.getAnnotation(ExcelClass.class);
		if (excelClass != null) {
			sheetName = excelClass.sheetName();
		}
		HSSFSheet sheet = wb.createSheet(sheetName);
		
		// 创建表头
		HSSFRow titleRow = sheet.createRow(0);
		HSSFCellStyle titleStyle = wb.createCellStyle();
		HSSFFont titleFont = wb.createFont(); // 表头字体
		titleFont.setBold(true);
		titleStyle.setFont(titleFont);
		titleStyle.setAlignment(HorizontalAlignment.CENTER);
		titleStyle.setBorderRight(BorderStyle.THIN); // 右边框
		titleStyle.setBorderBottom(BorderStyle.THICK); // 下边框
		titleStyle.setBorderLeft(BorderStyle.THIN); // 左边框
		titleStyle.setBorderTop(BorderStyle.THICK); // 上边框
		for (int index = 0; index < fields.length; index++) {
			ExcelField excelField = fields[index].getAnnotation(ExcelField.class);
			if (excelField != null) {
				fields[index].setAccessible(true);
				HSSFCell titleCell = titleRow.createCell(index);
				titleCell.setCellValue(excelField.titleName());
				titleCell.setCellStyle(titleStyle);
			}
		}
		
		// 创建数据行
		HSSFCellStyle dataStyle = wb.createCellStyle();
		dataStyle.setAlignment(HorizontalAlignment.CENTER);
		for (int iRow = 0; iRow < dataList.size(); iRow++) {
			HSSFRow dataRow = sheet.createRow(iRow + 1);
			// 填充数据列
			for (int iColumn = 0; iColumn < fields.length; iColumn++) {
				T dataObj = dataList.get(iRow);
				ExcelField excelField = fields[iColumn].getAnnotation(ExcelField.class);
				if (excelField != null) {
					HSSFCell dataCell = dataRow.createCell(iColumn);
					dataCell.setCellStyle(dataStyle);
					try {
						fields[iColumn].setAccessible(true);
						dataCell.setCellValue(fields[iColumn].get(dataObj) + "");
					} catch (Exception e) {
						// Ignore
					}
				}
			}
		}
		
		return wb;
	}
}
