package org.krproject.ocean.vitamins.admin.config;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.krproject.ocean.vitamins.admin.constants.AdminConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import com.fasterxml.classmate.ResolvedType;
import com.fasterxml.classmate.TypeResolver;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

import io.swagger.annotations.Api;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.ParameterBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.schema.ModelReference;
import springfox.documentation.schema.ResolvedTypes;
import springfox.documentation.schema.TypeNameExtractor;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.AuthorizationScope;
import springfox.documentation.service.GrantType;
import springfox.documentation.service.OAuth;
import springfox.documentation.service.Parameter;
import springfox.documentation.service.ResolvedMethodParameter;
import springfox.documentation.service.ResourceOwnerPasswordCredentialsGrant;
import springfox.documentation.service.SecurityReference;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.schema.contexts.ModelContext;
import springfox.documentation.spi.service.OperationBuilderPlugin;
import springfox.documentation.spi.service.contexts.OperationContext;
import springfox.documentation.spi.service.contexts.ParameterContext;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger.web.SecurityConfiguration;
import springfox.documentation.swagger.web.SecurityConfigurationBuilder;
import springfox.documentation.swagger2.annotations.EnableSwagger2;


/**
 * Admin Swagger 配置.
 * 
 * <p>OpenAPI 2.0 规格文档：
 * <a href="https://github.com/OAI/OpenAPI-Specification/blob/master/versions/2.0.md"> OpenAPI 2.0 </a>
 * 
 * <p>关于eclipse deprecated warning的处理:
 * <a href="https://github.com/springfox/springfox/issues/1307"> ClassOrApiAnnotationResourceGrouping deprecated </a>
 * 
 * @author Tiger
 */
@ConditionalOnProperty(prefix = "krproject.ocean.admin", name = "swagger-enabled", matchIfMissing = false)
@Configuration
@EnableSwagger2 // Enable swagger 2.0 spec
@EnableConfigurationProperties({AdminProperties.class, ServerProperties.class})
public class AdminSwaggerConfig  {

	@Resource
	private AdminProperties adminProperties;

	@Resource
	private ServerProperties serverProperties;
	
	@Autowired
	private ConfigurableApplicationContext context;
	
	@PostConstruct
	public void postConstruct() {
				
		// Creating Docket Dynamically per Rest Controller
		Map<String, Object> controllers = this.context.getBeansWithAnnotation(Api.class);
		for (Object obj : controllers.values()) {
			Api rm = ClassUtils.getUserClass(obj).getAnnotation(Api.class);
			if (rm != null && StringUtils.hasLength(rm.value())) {
				String groupName = rm.value();
				Docket docket = new Docket(DocumentationType.SWAGGER_2)
						.groupName(groupName).apiInfo(apiInfo())
						.forCodeGeneration(Boolean.TRUE).select()
						.paths(PathSelectors.regex(groupName + ".*")).build()
						.securitySchemes(Collections.singletonList(securitySchema()))
						.securityContexts(Collections.singletonList(securityContext()));
				this.context.getBeanFactory().registerSingleton(groupName + "_docket_api", docket);
			}
		}
	}
	
	@Bean
	public Docket defaultApi() {
		// Default Docket to show all.
		return new Docket(DocumentationType.SWAGGER_2)
				.apiInfo(apiInfo())
				.forCodeGeneration(Boolean.TRUE).select()
				.apis(RequestHandlerSelectors.withClassAnnotation(Api.class)).build()
				.securitySchemes(Collections.singletonList(securitySchema()))
				.securityContexts(Collections.singletonList(securityContext()));
	}
	
	@Bean
	public Docket oauthApi() {
		// Oauth Docket.
		String oauth2Group = "/oauth";
		return new Docket(DocumentationType.SWAGGER_2)
				.groupName(oauth2Group)
				.apiInfo(apiInfo())
				.forCodeGeneration(Boolean.TRUE).select()
				.paths(PathSelectors.regex(oauth2Group + ".*")).build()
				.securitySchemes(Collections.singletonList(securitySchema()))
				.securityContexts(Collections.singletonList(securityContext()));
	}

	@Bean
	public SecurityConfiguration securityInfo() {
		return SecurityConfigurationBuilder.builder().build();
	}
	
	private OAuth securitySchema() {
		List<AuthorizationScope> authorizationScopeList = new ArrayList<AuthorizationScope>();
//		authorizationScopeList.add(new AuthorizationScope("read", "read all"));
//		authorizationScopeList.add(new AuthorizationScope("trust", "trust all"));
//		authorizationScopeList.add(new AuthorizationScope("write", "access all"));

		List<GrantType> grantTypes = new ArrayList<GrantType>();
		String tokenUrl = "/oauth/token"; // spring security oauth2 default token path
		if (this.serverProperties.getServlet().getContextPath() != null) {
			tokenUrl = this.serverProperties.getServlet().getContextPath() + tokenUrl;
		}
		GrantType creGrant = new ResourceOwnerPasswordCredentialsGrant(tokenUrl);
		grantTypes.add(creGrant);

		return new OAuth(AdminConstants.OAUTH2_RESOURCE_ID, authorizationScopeList, grantTypes);
	}

	private SecurityContext securityContext() {
		return SecurityContext.builder().securityReferences(defaultAuth()).forPaths(PathSelectors.ant("/**")).build();
	}

	private List<SecurityReference> defaultAuth() {
		final AuthorizationScope[] authorizationScopes = new AuthorizationScope[3];
		authorizationScopes[0] = new AuthorizationScope("read", "read all");
		authorizationScopes[1] = new AuthorizationScope("trust", "trust all");
		authorizationScopes[2] = new AuthorizationScope("write", "write all");
		return Collections.singletonList(new SecurityReference(AdminConstants.OAUTH2_RESOURCE_ID, authorizationScopes));
	}

	
	private ApiInfo apiInfo() {
		return new ApiInfoBuilder()
				.title("Restful Api Documentation")
				.description("<p>Restful接口文档，采用Oauth2 Password认证模式。</p>"
						+ "<p>提示：上面的链接为符合"
						+ "<a href=\"https://github.com/OAI/OpenAPI-Specification/blob/master/versions/2.0.md\"> OpenAPI 2.0 规范</a>"
						+ "的json格式接口文档，可以保存到本地后，导入Postman等工具用于测试！</p>"
						+ "<p><b>注意：本功能页面应仅用于开发测试环境，生产环境应关闭本功能页面！</b></p>")
				.build();
	}
	
	/**
	 * 分页的Swagger转换插件.
	 * 
	 * <p>参考：<a href="https://github.com/springfox/springfox/issues/755"> Pageable solution for springfox </a>
	 * 
	 * @author Tiger
	 */
	@Component
	@Order(Ordered.LOWEST_PRECEDENCE)
	public class AdminSwaggerPageablePlugin implements OperationBuilderPlugin {
		private final TypeNameExtractor nameExtractor;
		private final TypeResolver resolver;
		private final ResolvedType pageableType;
		
		@Autowired
		public AdminSwaggerPageablePlugin(TypeNameExtractor nameExtractor, TypeResolver resolver) {
			this.nameExtractor = nameExtractor;
			this.resolver = resolver;
			this.pageableType = resolver.resolve(Pageable.class);
		}
		
		@Override
		public void apply(OperationContext context) {
			List<ResolvedMethodParameter> methodParameters = context.getParameters();
			List<Parameter> parameters = Lists.newArrayList();
		
			for (ResolvedMethodParameter methodParameter : methodParameters) {
				ResolvedType resolvedType = methodParameter.getParameterType();
			
				if (this.pageableType.equals(resolvedType)) {
					ParameterContext parameterContext = new ParameterContext(methodParameter,
						new ParameterBuilder(),
						context.getDocumentationContext(),
						context.getGenericsNamingStrategy(),
						context);
					Function<ResolvedType, ? extends ModelReference> factory = createModelRefFactory(parameterContext);
			
					ModelReference intModel = factory.apply(this.resolver.resolve(Integer.TYPE));
					ModelReference stringModel = factory.apply(this.resolver.resolve(List.class, String.class));
			
					parameters.add(new ParameterBuilder()
						.parameterType("query")
						.name("page")
						.modelRef(intModel)
						.description("Results page you want to retrieve (0..N)").build());
					parameters.add(new ParameterBuilder()
						.parameterType("query")
						.name("size")
						.modelRef(intModel)
						.description("Number of records per page").build());
					parameters.add(new ParameterBuilder()
						.parameterType("query")
						.name("sort")
						.modelRef(stringModel)
						.allowMultiple(true)
						.description("Sorting criteria in the format: property(,asc|desc). "
							+ "Default sort order is ascending. "
							+ "Multiple sort criteria are supported.")
						.build());
					context.operationBuilder().parameters(parameters);
				}
			}
		}
		
		@Override
		public boolean supports(DocumentationType delimiter) {
			return true;
		}
		
		private Function<ResolvedType, ? extends ModelReference> createModelRefFactory(ParameterContext context) {
			ModelContext modelContext = ModelContext.inputParam(
					context.getGroupName(),
					context.resolvedMethodParameter().getParameterType(),
					context.getDocumentationType(),
					context.getAlternateTypeProvider(),
					context.getGenericNamingStrategy(),
					context.getIgnorableParameterTypes());
			return ResolvedTypes.modelRefFactory(modelContext, this.nameExtractor);
		}
	}
}
