package org.krproject.ocean.vitamins.admin.config;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.krproject.ocean.vitamins.admin.constants.AdminConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpMethod;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.config.annotation.web.configuration.EnableResourceServer;
import org.springframework.security.oauth2.config.annotation.web.configuration.ResourceServerConfigurerAdapter;
import org.springframework.security.oauth2.config.annotation.web.configurers.ResourceServerSecurityConfigurer;
import org.springframework.security.oauth2.provider.token.TokenStore;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.access.AccessDeniedHandler;


/**
 * OAuth2资源服务配置.
 * 
 * @author Tiger
 */
@Configuration
@EnableResourceServer
public class AdminOAuth2ResourceConfig extends ResourceServerConfigurerAdapter {
	
	@Autowired
	private TokenStore tokenStore;
	
	@Override
	public void configure(ResourceServerSecurityConfigurer resource) {
		resource.resourceId(AdminConstants.OAUTH2_RESOURCE_ID).tokenStore(this.tokenStore);
	}
	
	@Override
	public void configure(HttpSecurity http) throws Exception {
		http.csrf().disable().formLogin().disable()
			.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS).and()
			.exceptionHandling()
				.authenticationEntryPoint(new AuthExceptionEntryPoint())
				.accessDeniedHandler(new AccessExceptionEntryPoint())
			.and()
			.authorizeRequests()
				.antMatchers(HttpMethod.OPTIONS).permitAll()
				.anyRequest().authenticated();
	}
	
	
	/**
	 * 异常处理，认证失败返回401.
	 *
	 * @author Tiger
	 */
	public class AuthExceptionEntryPoint implements AuthenticationEntryPoint {

		@Override
		public void commence(HttpServletRequest request, HttpServletResponse response,
				AuthenticationException authException) throws IOException, ServletException {
			response.sendError(HttpServletResponse.SC_UNAUTHORIZED, authException.getMessage());
		}
	}
	
	
	/**
	 * 异常处理，鉴权失败返回403.
	 *
	 * @author Tiger
	 */
	public class AccessExceptionEntryPoint implements AccessDeniedHandler {
		
		@Override
		public void handle(HttpServletRequest request, HttpServletResponse response,
				AccessDeniedException accessDeniedException) throws IOException, ServletException {
			response.sendError(HttpServletResponse.SC_FORBIDDEN, accessDeniedException.getMessage());
		}
	}

}
