package org.krproject.ocean.skeletons.octopus.online.transaction;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.krproject.ocean.skeletons.octopus.online.constants.OctopusSkeletonOnlineConstants;
import org.krproject.ocean.skeletons.octopus.online.transaction.event.ReverseCompletedEvent;
import org.krproject.ocean.skeletons.octopus.online.transaction.model.OnlineOutboundModel;
import org.krproject.ocean.skeletons.octopus.online.transaction.service.OctopusOnlineTransactionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.integration.annotation.MessageEndpoint;
import org.springframework.integration.annotation.ServiceActivator;

import lombok.extern.slf4j.Slf4j;


/**
 * 冲正类交易服务端点.
 * 
 * @author Tiger
 *
 */
@Slf4j
@MessageEndpoint
public class ReverseServiceActivator {

	@Resource
	private ApplicationContext applicationContext;

	@Resource
	private OctopusOnlineTransactionService octopusOnlineJournalService;

	@Autowired(required = false)
	private List<AbstractReverser> reversers;
	
	private Map<String, AbstractReverser> reverserMap = new ConcurrentHashMap<String, AbstractReverser>();
	
	@PostConstruct
	public void init() {
		if (this.reversers != null) {
			for (AbstractReverser reverser : this.reversers) {
				AbstractReverser reverserFound = this.reverserMap.get(reverser.getProcessorName());
				if (reverserFound != null) {
					log.error("reverser {} and {} got same processor name：{}", reverserFound, reverser, reverser.getProcessorName());
					throw new RuntimeException("duplicate reverser for processor " + reverser.getProcessorName());
				}
				this.reverserMap.put(reverser.getProcessorName(), reverser);
			}
		}
	}
	
	
	@ServiceActivator(inputChannel = OctopusSkeletonOnlineConstants.REVERSE_CHANNEL_NAME)
	private void activate(String inboundId) throws Exception {
		log.info("reverse inboundId:{} ", inboundId);
		
		Boolean reversed = true; // 本交易是否已冲正
		// 根据inbound id逆序遍历所有的outbound流水，并冲正所有交易
		for (OnlineOutboundModel outbound : this.octopusOnlineJournalService.listOutboundForReverse(inboundId)) {
			log.debug("reverse outboundId:{} of inboundId:{}!", outbound.getOutboundId(), inboundId);

			// 根据处理器找到对应冲正器
			AbstractReverser reverser = this.reverserMap.get(outbound.getProcessorName());
			if (reverser != null) {
				// 使用冲正器对原Outbound交易冲正
				Boolean result = reverseOutbound(outbound.getOutboundId(), reverser);
				if (result == null) {
					reversed = null;
					break; // 冲正异常
				} else if (!result) {
					reversed = false;
					break; // 冲正失败
				}
			}
		}
		
		log.info("reverse inboundId:{} result:{}", inboundId, reversed);
		
		// 广播冲正完成事件
		this.applicationContext.publishEvent(new ReverseCompletedEvent(this, inboundId, reversed));
	}


	/**
	 * 冲正下游系统交易.
	 * @param outboundId 下游系统交易编号
	 * @param reverser 冲正器
	 * @return 交易是否已冲正，null-冲正异常，true-冲正成功，false-冲正失败
	 */
	private Boolean reverseOutbound(String outboundId, AbstractReverser reverser) {
		Boolean result = null; //本交易是否已冲正
		
		// 对原交易冲正，有间隔与次数限制
		for (int retry = 1; retry <= reverser.getReverseTimes(); retry++) {
			log.debug("reverse outboundId:{} with times:{}!", outboundId, retry);
							
			// 调用冲正器进行冲正处理
			try {
				if (reverser.reverse(outboundId)) {
					log.debug("reverse outboundId:{} success!", outboundId);
					result = true;
				} else {
					log.debug("reverse outboundId:{} failure!", outboundId);
					result = false;
				}
				break;  // 有明确结果，不再冲正
			} catch (Exception e) {
				log.debug("reverse outboundId:{} exception:{}!, retry later!", outboundId, e);
				try {
					// 睡眠质询间隔后，继续重试
					Thread.sleep(reverser.getReverseInterval());
				} catch (InterruptedException e1) {
					// ignore
				}
			}
		}
		return result;
	}
}
