package org.krproject.ocean.skeletons.octopus.batch.endpoint;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import org.krproject.ocean.skeletons.octopus.batch.api.OctopusBatchRequest;
import org.krproject.ocean.skeletons.octopus.batch.api.OctopusBatchResponse;
import org.krproject.ocean.skeletons.octopus.batch.event.OctopusBatchCompletedEvent;
import org.krproject.ocean.skeletons.octopus.batch.exception.OctopusSkeletonBatchBadRequestException;
import org.krproject.ocean.skeletons.octopus.batch.exception.OctopusSkeletonBatchException;
import org.krproject.ocean.vitamins.batch.exception.BatchException;
import org.krproject.ocean.vitamins.service.config.ServiceProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.util.ClassUtils;

import lombok.extern.slf4j.Slf4j;


/**
 * 批量服务端点抽象类.
 * 
 * @param <REQ> 请求泛型
 * @param <RESP> 响应泛型
 * @author Tiger
 *
 */
@Slf4j
public abstract class AbstractOctopusBatchActivator<REQ extends OctopusBatchRequest<RESP>, RESP extends OctopusBatchResponse> {
	
	@Autowired
	private Validator validator;

	@Autowired(required = false)
	private List<AbstractOctopusBatchHandler<?, ?>> handlers;
	
	@Autowired
	private ServiceProperties serviceProperties;
	
	@Resource
	private ApplicationContext applicationContext;

	private static boolean initialized = false;

	private static final Map<String, AbstractOctopusBatchHandler<?, ?>> HANDLER_MAP = new ConcurrentHashMap<String, AbstractOctopusBatchHandler<?, ?>>();
	
	@PostConstruct
	public void init() {
		if (!initialized) {
			if (this.handlers != null) {
				for (AbstractOctopusBatchHandler<?, ?> handler : this.handlers) {
					AbstractOctopusBatchHandler<?, ?> handlerFound = HANDLER_MAP.get(handler.getRequestClass().getName());
					if (handlerFound != null) {
						log.error("handler {} and {} got same request name：{}", 
								handlerFound, handler, handler.getRequestClass().getName());
						throw new RuntimeException("duplicate handler for request " + handler.getRequestClass().getName());
					}
					HANDLER_MAP.put(handler.getRequestClass().getName(), handler);
					log.debug("Loaded Handler:{} for Request:{}", ClassUtils.getUserClass(handler).getName(), handler.getRequestClass().getName());
				}
			}
			initialized = true;
		}
	}
	
	/**
	 * 根据异常获取响应.
	 * 
	 * @param request 请求
	 * @param exception 异常
	 * @return 响应
	 */
	public abstract RESP responseWithException(REQ request, Exception exception);

	
	/**
	 * 日志表插入.
	 * 
	 * @param request 请求
	 * 
	 * @return 日志对象
	 */
	public abstract Object insertJournal(REQ request);
	
	
	/**
	 * 日志表更新.
	 * 
	 * @param journal 日志
	 * @param handler 处理器
	 * @param response 响应
	 */
	public abstract void updateJournal(Object journal, AbstractOctopusBatchHandler<REQ, RESP> handler, RESP response);

	
	@SuppressWarnings("unchecked")
	protected RESP doActivate(REQ request) throws BatchException {
		
		// 请求不能为空
		if (request == null) {
			log.error("request cant't be null");
			throw new OctopusSkeletonBatchBadRequestException("request cant't be null");
		}
		
		// 获取请求时间
		long requestTime = System.currentTimeMillis(); 
		
		// 请求的合法性校验.
		Set<ConstraintViolation<REQ>> constraintViolations = this.validator.validate(request);
		for (ConstraintViolation<REQ> cv: constraintViolations) {
			log.error("错误描述:{}, {}", cv.getPropertyPath(), cv.getMessage());
			throw new OctopusSkeletonBatchBadRequestException(cv.getMessage());
		}
		
		// 记录Journal日志
		Object journal = insertJournal(request);
		
		RESP response = null;
		AbstractOctopusBatchHandler<REQ, RESP> handler = null;
		try {
			// 根据请求类获取对应的handler
			String requestClassName = ClassUtils.getUserClass(request).getName();
			handler = (AbstractOctopusBatchHandler<REQ, RESP>) HANDLER_MAP.get(requestClassName);
			if (handler == null) {
				log.warn("Fail to get handler for request:{}", requestClassName);
				throw new OctopusSkeletonBatchException("No handler for " + requestClassName);
			}
			log.debug("Found Handler:{} for request:{}", handler.getClass(), request.getClass());

			// 调用处理器方法处理
			response = handler.handle(request);			
		} catch (Exception exception) {
			response = responseWithException(request, exception);
		}

		// 更新journal日志
		updateJournal(journal, handler, response);
		
		// 获取响应时间
		long responseTime = System.currentTimeMillis(); 
		
		// 广播批量交易处理完成事件
		this.applicationContext.publishEvent(new OctopusBatchCompletedEvent(this, this.serviceProperties.getInstanceId(), 
				request, new Date(requestTime), handler, response, new Date(responseTime)));
		
		return response;
	}
}
