/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.collection.pool;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.LongSupplier;
import org.junit.Assert;
import org.junit.Test;
import org.neo4j.collection.pool.LinkedQueuePool;

public class LinkedQueuePoolTest {
    @Test
    public void shouldTimeoutGracefully() {
        FakeClock clock = new FakeClock();
        LinkedQueuePool.CheckStrategy.TimeoutCheckStrategy timeStrategy = new LinkedQueuePool.CheckStrategy.TimeoutCheckStrategy(100L, (LongSupplier)clock);
        while (clock.getAsLong() <= 100L) {
            Assert.assertFalse((boolean)timeStrategy.shouldCheck());
            clock.forward(10L, TimeUnit.MILLISECONDS);
        }
        Assert.assertTrue((boolean)timeStrategy.shouldCheck());
        clock.forward(1L, TimeUnit.MILLISECONDS);
        Assert.assertFalse((boolean)timeStrategy.shouldCheck());
    }

    @Test
    public void shouldBuildUpGracefullyUntilReachedMinPoolSize() {
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 5);
        List<FlyweightHolder<Object>> flyweightHolders = this.acquireFromPool(pool, 5);
        for (FlyweightHolder<Object> flyweightHolder : flyweightHolders) {
            flyweightHolder.release();
        }
        Assert.assertEquals((long)-1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)-1L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldBuildUpGracefullyWhilePassingMinPoolSizeBeforeTimerRings() {
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 5);
        List<FlyweightHolder<Object>> flyweightHolders = this.acquireFromPool(pool, 15);
        for (FlyweightHolder<Object> flyweightHolder : flyweightHolders) {
            flyweightHolder.release();
        }
        Assert.assertEquals((long)-1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)-1L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)15L, (long)stateMonitor.created.get());
        Assert.assertEquals((long)10L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldUpdateTargetSizeWhenSpikesOccur() {
        int MIN_SIZE = 5;
        int MAX_SIZE = 10;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 5);
        List<FlyweightHolder<Object>> holders = this.acquireFromPool(pool, 10);
        clock.forward(110L, TimeUnit.MILLISECONDS);
        holders.addAll(this.acquireFromPool(pool, 1));
        Assert.assertEquals((long)11L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)11L, (long)stateMonitor.targetSize.get());
    }

    @Test
    public void shouldKeepSmallPeakAndNeverDisposeIfAcquireAndReleaseContinuously() {
        boolean MIN_SIZE = true;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 1);
        for (int i = 0; i < 200; ++i) {
            List<FlyweightHolder<Object>> newOnes = this.acquireFromPool(pool, 1);
            for (FlyweightHolder<Object> newOne : newOnes) {
                newOne.release();
            }
        }
        Assert.assertEquals((long)-1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)1L, (long)stateMonitor.created.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldSlowlyReduceTheNumberOfFlyweightsInThePoolWhenFlyweightsAreReleased() {
        int MIN_SIZE = 50;
        int MAX_SIZE = 200;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 50);
        LinkedList<FlyweightHolder<Object>> holders = new LinkedList<FlyweightHolder<Object>>();
        this.buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(200, clock, pool, holders);
        clock.forward(110L, TimeUnit.MILLISECONDS);
        for (int i = 0; i < 200; ++i) {
            this.acquireFromPool(pool, 1).get(0).release();
        }
        Assert.assertEquals((long)1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)50L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)151L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldMaintainPoolAtHighWatermarkWhenConcurrentUsagePassesMinSize() {
        int MIN_SIZE = 50;
        int MAX_SIZE = 200;
        int MID_SIZE = 90;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 50);
        LinkedList<FlyweightHolder<Object>> holders = new LinkedList<FlyweightHolder<Object>>();
        this.buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(200, clock, pool, holders);
        Assert.assertEquals((long)201L, (long)stateMonitor.currentPeakSize.get());
        for (int i = 0; i < 2; ++i) {
            clock.forward(110L, TimeUnit.MILLISECONDS);
            for (FlyweightHolder<Object> holder : this.acquireFromPool(pool, 90)) {
                holder.release();
            }
            clock.forward(110L, TimeUnit.MILLISECONDS);
            for (FlyweightHolder<Object> holder : this.acquireFromPool(pool, 90)) {
                holder.release();
            }
        }
        Assert.assertEquals((long)90L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)90L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)111L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldReclaimAndRecreateWhenLullBetweenSpikesOccurs() {
        int MIN_SIZE = 50;
        int BELOW_MIN_SIZE = 10;
        int MAX_SIZE = 200;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 50);
        LinkedList<FlyweightHolder<Object>> holders = new LinkedList<FlyweightHolder<Object>>();
        this.buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(200, clock, pool, holders);
        clock.forward(110L, TimeUnit.MILLISECONDS);
        for (int i = 0; i < 30; ++i) {
            for (FlyweightHolder<Object> holder : this.acquireFromPool(pool, 10)) {
                holder.release();
            }
            clock.forward(110L, TimeUnit.MILLISECONDS);
        }
        Assert.assertTrue((String)("Expected " + stateMonitor.currentPeakSize.get() + " <= " + 10), (stateMonitor.currentPeakSize.get() <= 10 ? 1 : 0) != 0);
        Assert.assertEquals((long)50L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)151L, (long)stateMonitor.disposed.get());
        stateMonitor.created.set(0);
        stateMonitor.disposed.set(0);
        holders.addAll(this.acquireFromPool(pool, 200));
        Assert.assertEquals((long)150L, (long)stateMonitor.created.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    private void buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(int MAX_SIZE, FakeClock clock, LinkedQueuePool<Object> pool, List<FlyweightHolder<Object>> holders) {
        holders.addAll(this.acquireFromPool(pool, MAX_SIZE));
        clock.forward(110L, TimeUnit.MILLISECONDS);
        holders.addAll(this.acquireFromPool(pool, 1));
        for (FlyweightHolder<Object> holder : holders) {
            holder.release();
        }
    }

    private LinkedQueuePool<Object> getLinkedQueuePool(StatefulMonitor stateMonitor, FakeClock clock, int minSize) {
        return new LinkedQueuePool(minSize, Object::new, (LinkedQueuePool.CheckStrategy)new LinkedQueuePool.CheckStrategy.TimeoutCheckStrategy(100L, (LongSupplier)clock), (LinkedQueuePool.Monitor)stateMonitor);
    }

    private <R> List<FlyweightHolder<R>> acquireFromPool(LinkedQueuePool<R> pool, int times) {
        LinkedList<FlyweightHolder<R>> acquirers = new LinkedList<FlyweightHolder<R>>();
        for (int i = 0; i < times; ++i) {
            FlyweightHolder holder = new FlyweightHolder(pool);
            acquirers.add(holder);
            holder.run();
        }
        return acquirers;
    }

    private static class FakeClock
    implements LongSupplier {
        private long time;

        private FakeClock() {
        }

        @Override
        public long getAsLong() {
            return this.time;
        }

        public void forward(long amount, TimeUnit timeUnit) {
            this.time += timeUnit.toMillis(amount);
        }
    }

    private static class StatefulMonitor
    implements LinkedQueuePool.Monitor<Object> {
        public AtomicInteger currentPeakSize = new AtomicInteger(-1);
        public AtomicInteger targetSize = new AtomicInteger(-1);
        public AtomicInteger created = new AtomicInteger(0);
        public AtomicInteger acquired = new AtomicInteger(0);
        public AtomicInteger disposed = new AtomicInteger(0);

        private StatefulMonitor() {
        }

        public void updatedCurrentPeakSize(int currentPeakSize) {
            this.currentPeakSize.set(currentPeakSize);
        }

        public void updatedTargetSize(int targetSize) {
            this.targetSize.set(targetSize);
        }

        public void created(Object Object2) {
            this.created.incrementAndGet();
        }

        public void acquired(Object Object2) {
            this.acquired.incrementAndGet();
        }

        public void disposed(Object Object2) {
            this.disposed.incrementAndGet();
        }
    }

    private static class FlyweightHolder<R>
    implements Runnable {
        private final LinkedQueuePool<R> pool;
        private R resource;

        private FlyweightHolder(LinkedQueuePool<R> pool) {
            this.pool = pool;
        }

        @Override
        public void run() {
            this.resource = this.pool.acquire();
        }

        public void release() {
            this.pool.release(this.resource);
        }
    }
}

