/*******************************************************************************
 * Copyright (c) Faktor Zehn GmbH - faktorzehn.org
 * 
 * This source code is available under the terms of the AGPL Affero General Public License version
 * 3.
 * 
 * Please see LICENSE.txt for full license terms, including the additional permissions and
 * restrictions as well as the possibility of alternative license terms.
 *******************************************************************************/

package org.faktorips.runtime.productdataprovider.ejbclient;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.ejb.EJBException;
import javax.ejb.NoSuchEJBException;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.faktorips.productdataservice.IProductDataService;
import org.faktorips.productdataservice.XmlTimestampData;
import org.faktorips.runtime.IVersionChecker;
import org.faktorips.runtime.internal.DateTime;
import org.faktorips.runtime.internal.toc.CustomTocEntryObject;
import org.faktorips.runtime.internal.toc.EnumContentTocEntry;
import org.faktorips.runtime.internal.toc.GenerationTocEntry;
import org.faktorips.runtime.internal.toc.IReadonlyTableOfContents;
import org.faktorips.runtime.internal.toc.ProductCmptTocEntry;
import org.faktorips.runtime.internal.toc.ReadonlyTableOfContents;
import org.faktorips.runtime.internal.toc.TableContentTocEntry;
import org.faktorips.runtime.internal.toc.TestCaseTocEntry;
import org.faktorips.runtime.productdataprovider.AbstractProductDataProvider;
import org.faktorips.runtime.productdataprovider.DataModifiedException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * This product data provider loads the product data content from an ejb stateless session bean. The
 * service is called for every request of product data. The service provides the xml content with
 * the version of the actually loaded toc. This product data provider checks whether the table of
 * content loaded by the client is the same as used by the service. When the version differs, a
 * {@link DataModifiedException} is thrown by any method providing product data. This class holds
 * the {@link #toc} and the {@link #productDataService} in final variables. So you cannot reload
 * this provider but creating a new. We do so to avoid several concurrency problems.
 * 
 * @author dirmeier
 */
public class EjbProductDataProvider extends AbstractProductDataProvider {

    public static final String EJB_EXCEPTION_MESSAGE = "Exception while accessing product data service, while getting ";
    public static final String EJB_EXCEPTION_MESSAGE_CAUSED_BY = "Exception caused by ";

    /**
     * The version checker is not really used in this implementation. The problem is, that the
     * application server throws an {@link NoSuchEJBException} when a new version of the product
     * data service was deployed. That's why we always have to create a new
     * {@link EjbProductDataProvider} after any deployment.
     */
    private static IVersionChecker versionChecker = new IVersionChecker() {

        @Override
        public boolean isCompatibleVersion(String oldVersion, String newVersion) {
            return oldVersion.equals(newVersion);
        }
    };

    private final IProductDataService productDataService;

    private final ReadonlyTableOfContents toc;

    /**
     * This is the version of this product data provider. This version never change until a new
     * provider is created.
     */
    private final String version;

    /**
     * Initializing the product data provider getting information from the stateless session bean
     * <code>org.faktorips.productdataservice.ProductDataService</code>
     * 
     * The properties are used to initiate the {@link InitialContext} you have to set at least the
     * factory by property key Context.INITIAL_CONTEXT_FACTORY
     * 
     * Note: To avoid concurrency problems we do not reload the {@link #productDataService} in case
     * of any {@link EJBException}. The disadvantage of this purpose is, that the
     * {@link IVersionChecker} is getting nearly useless. In fact we have to reload the repository
     * whenever a new version is deployed.
     */
    EjbProductDataProvider(String beanName, InitialContext initialContext) {
        super(versionChecker);
        try {
            productDataService = (IProductDataService)initialContext.lookup(beanName);
        } catch (NamingException e) {
            throw new RuntimeException(e);
        }
        toc = loadToc();
        version = getBaseVersion();
    }

    /**
     * Use this constructor to create a new {@link EjbProductDataProvider} with an already
     * instantiated {@link IProductDataService}. For example if you have a custom lookup. Otherwise
     * use {@link EjbProductDataProviderFactory} to call the default {@link InitialContext} lookup.
     * 
     * @param productDataService The product data service you want to be used by this provider
     */
    public EjbProductDataProvider(IProductDataService productDataService) {
        super(versionChecker);
        if (productDataService == null) {
            throw new NullPointerException("Product data service must not be null");
        }
        this.productDataService = productDataService;
        toc = loadToc();
        version = getBaseVersion();
    }

    private ReadonlyTableOfContents loadToc() {
        XmlTimestampData timestampData = productDataService.getTocData();
        try {
            Document doc = getDocumentBuilder().parse(
                    new InputSource(new ByteArrayInputStream(timestampData.getXmlData())));
            Element tocElement = doc.getDocumentElement();
            ReadonlyTableOfContents readOnlyToc = new ReadonlyTableOfContents();
            readOnlyToc.initFromXml(tocElement);
            return readOnlyToc;
        } catch (IOException e) {
            throw new RuntimeException(e);
        } catch (SAXException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getVersion() {
        return version;
    }

    @Override
    public InputStream getEnumContentAsStream(EnumContentTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData;
        try {
            timestampData = productDataService.getEnumContent(tocEntry.getImplementationClassName());
        } catch (EJBException e) {
            throw createEjbException(tocEntry.getIpsObjectQualifiedName(), e);
        }
        throwExceptionIfExpired(tocEntry.getImplementationClassName(), timestampData.getVersion());
        return new ByteArrayInputStream(timestampData.getXmlData());
    }

    @Override
    public Element getProductCmptData(ProductCmptTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData;
        try {
            timestampData = productDataService.getProductCmptData(tocEntry.getIpsObjectId());
        } catch (EJBException e) {
            throw createEjbException(tocEntry.getIpsObjectQualifiedName(), e);
        }
        throwExceptionIfExpired(tocEntry.getIpsObjectId(), timestampData.getVersion());
        try {
            Document doc = getDocumentBuilder().parse(
                    new InputSource(new ByteArrayInputStream(timestampData.getXmlData())));
            return doc.getDocumentElement();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } catch (SAXException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Element getProductCmptGenerationData(GenerationTocEntry tocEntry) throws DataModifiedException {
        Element docElement = getProductCmptData(tocEntry.getParent());
        NodeList nl = docElement.getChildNodes();
        DateTime validFrom = tocEntry.getValidFrom();
        for (int i = 0; i < nl.getLength(); i++) {
            if ("Generation".equals(nl.item(i).getNodeName())) {
                Element genElement = (Element)nl.item(i);
                DateTime generationValidFrom = DateTime.parseIso(genElement.getAttribute("validFrom"));
                if (validFrom.equals(generationValidFrom)) {
                    return genElement;
                }
            }
        }
        throw new RuntimeException("Can't find the generation for the toc entry " + tocEntry);
    }

    @Override
    public InputStream getTableContentAsStream(TableContentTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData;
        try {
            timestampData = productDataService.getTableContent(tocEntry.getIpsObjectQualifiedName());
        } catch (EJBException e) {
            throw createEjbException(tocEntry.getIpsObjectQualifiedName(), e);
        }
        throwExceptionIfExpired(tocEntry.getIpsObjectQualifiedName(), timestampData.getVersion());
        return new ByteArrayInputStream(timestampData.getXmlData());
    }

    @Override
    public Element getTestcaseElement(TestCaseTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData;
        try {
            timestampData = productDataService.getTestCaseData(tocEntry.getIpsObjectQualifiedName());
        } catch (EJBException e) {
            throw createEjbException(tocEntry.getIpsObjectQualifiedName(), e);
        }

        throwExceptionIfExpired(tocEntry.getIpsObjectId(), timestampData.getVersion());
        try {
            Document doc = getDocumentBuilder().parse(
                    new InputSource(new ByteArrayInputStream(timestampData.getXmlData())));
            return doc.getDocumentElement();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } catch (SAXException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public synchronized IReadonlyTableOfContents getToc() {
        return toc;
    }

    @Override
    public boolean isCompatibleToBaseVersion() {
        try {
            return super.isCompatibleToBaseVersion();
        } catch (EJBException exception) {
            return false;
        }
    }

    @Override
    public String getBaseVersion() {
        return productDataService.getProductDataVersion();
    }

    private void throwExceptionIfExpired(String ipsObject, String version) throws DataModifiedException {
        if (!getVersionChecker().isCompatibleVersion(getVersion(), version)) {
            DataModifiedException e = new DataModifiedException(MODIFIED_EXCEPTION_MESSAGE + ipsObject, this.version,
                    version);
            throw e;
        }
    }

    /**
     * In case of an {@link EJBException} we create a {@link DataModifiedException} telling the
     * client that something was wrong with the ejb. In most cases the bean was undeployed e.g.
     * because a new version was deployed.
     * 
     * @param e the {@link EJBException} to handle
     * @return a DataModifiedException
     */
    private DataModifiedException createEjbException(String objectName, EJBException e) {
        String causedByExceptionMsg = getEJBExceptionCauseMessage(e);
        DataModifiedException dataModifiedException = new DataModifiedException(EJB_EXCEPTION_MESSAGE_CAUSED_BY
                + causedByExceptionMsg + " ," + objectName, version, "<unknown>");

        Exception causedByException = e.getCausedByException();
        if (causedByException != null) {
            dataModifiedException.initCause(causedByException);
        } else {
            dataModifiedException.initCause(e);
        }

        return dataModifiedException;
    }

    private String getEJBExceptionCauseMessage(EJBException e) {
        Exception causedByException = e.getCausedByException();
        if (causedByException != null) {
            return causedByException.getMessage();
        }
        Throwable cause = e.getCause();
        if (cause != null) {
            return cause.getMessage();
        }
        return "<unknown>";
    }

    @Override
    public <T> Element getTocEntryData(CustomTocEntryObject<T> tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData;
        try {
            timestampData = productDataService.getTocEntryData(tocEntry.getRuntimeObjectClass(),
                    tocEntry.getIpsObjectQualifiedName());
        } catch (EJBException e) {
            throw createEjbException(tocEntry.getIpsObjectQualifiedName(), e);
        }

        throwExceptionIfExpired(tocEntry.getIpsObjectId(), timestampData.getVersion());
        try {
            Document doc = getDocumentBuilder().parse(
                    new InputSource(new ByteArrayInputStream(timestampData.getXmlData())));
            return doc.getDocumentElement();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } catch (SAXException e) {
            throw new RuntimeException(e);
        }
    }
}
