/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.content;

import org.apache.log4j.Logger;
import org.dspace.core.Context;
import org.dspace.core.LogManager;
import org.dspace.storage.bitstore.BitstreamStorageManager;
import org.dspace.storage.rdbms.DatabaseManager;
import org.dspace.storage.rdbms.TableRow;
import org.dspace.storage.rdbms.TableRowIterator;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Date;

/**
 * BitstreamUtil Class is used to hold static lookup methods and avoid overriding Bitstream directly.
 *
 * @author Mark Diggory.
 */
public class BitstreamUtil {

    /** log4j category */
    private static Logger log = Logger.getLogger(BitstreamUtil.class);

    /**
     * Allow access to delete method on Bitstream from code outside the package.
     * @param bitstream
     * @throws SQLException
     */
    public static void delete(Context context, Bitstream bitstream, boolean cleanup) throws SQLException, IOException {
        bitstream.delete();
        if(cleanup)
            BitstreamStorageManager.cleanup(context, bitstream.getID(), true, true);
    }

    /**
     * Allow access to delete method on Bitstream from code outside the package.
     * @param bitstream
     * @throws SQLException
     */
    public static void delete(Bitstream bitstream) throws SQLException {
        bitstream.delete();
    }

    /**
     * isDeleted should have been a public method on Bitstream, utility method to determine if Bitstream is deleted.
     * @param bitstream
     * @return
     * @throws SQLException
     */
    public static boolean isDeleted(Bitstream bitstream) throws SQLException {
        return bitstream.isDeleted();
    }

    /**
     * Add support to retrieve the last modified date. Ideally, this should be moved inside Bitstream class in
     * future DSpace version.
     *
     * @param context
     * @param uuid
     * @return
     * @throws SQLException
     */
    public static Bitstream findByUuid(Context context, String uuid) throws SQLException{

        if(uuid.contains(":"))
            uuid = uuid.substring(uuid.lastIndexOf(":")+1);

        TableRow row = DatabaseManager.findByUnique(context, "bitstream", "uuid", uuid);

        if (row == null)
        {
            if (log.isDebugEnabled())
            {
                log.debug(LogManager.getHeader(context, "find_bitstream",
                        "not_found,bitstream_id=" + uuid));
            }

            return null;
        }

        // not null, return Bitstream
        if (log.isDebugEnabled())
        {
            log.debug(LogManager.getHeader(context, "find_bitstream",
                    "uuid=" + uuid));
        }

        // First check the cache
        Bitstream fromCache = (Bitstream) context
                .fromCache(Bitstream.class, row.getIntColumn("bitstream_id"));

        if (fromCache != null)
        {
            return fromCache;
        }
        else
        {
            return new Bitstream(context, row);
        }

    }

    /**
     *
     * @param context
     * @param bitstream
     * @return
     * @throws SQLException
     */
    public static String getUuid(Context context, Bitstream bitstream) throws SQLException{

        TableRowIterator row = DatabaseManager.query(context,"select * from bitstream where bitstream_id = "+bitstream.getID());

        if (row.hasNext())
        {
            return row.next().getStringColumn("uuid");
        }
        return null;
    }

    /**
     * Add support to retrieve the last modified date. Ideally, this should be moved inside Bitstream class in
     * future DSpace version.
     *
     * @param context
     * @param bitstream
     * @return
     * @throws SQLException
     */
    public static Date getLastModifiedDate(Context context,Bitstream bitstream) throws SQLException{

        TableRowIterator row = DatabaseManager.query(context,"select * from bitstream where bitstream_id = "+bitstream.getID());

        if (row.hasNext())
        {
            return row.next().getDateColumn("last_modified_date");
        }
        return null;
    }

    /**
     * Add support to get the date created from the Bitstream. Ideally, this should be moved inside Bitstream class in
     * future DSpace version.
     *
     * @param context
     * @param bitstream
     * @return
     * @throws SQLException
     */
    public static Date getDateCreated(Context context, Bitstream bitstream) throws SQLException {

        TableRowIterator row = DatabaseManager.query(context,"select * from bitstream where bitstream_id = "+bitstream.getID());

        if (row.hasNext())
        {
            return row.next().getDateColumn("create_date");
        }
        return null;
    }


}
