package org.criticalsection.maven.scm.provider.monotone.command.status;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.command.status.AbstractStatusCommand;
import org.apache.maven.scm.command.status.StatusScmResult;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFile;
import org.apache.maven.scm.ScmFileStatus;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.codehaus.plexus.util.StringUtils;
import org.criticalsection.mtn4java.commands.Command;
import org.criticalsection.mtn4java.commands.automate.InventoryCommand;
import org.criticalsection.mtn4java.commands.automate.InventoryInformation;
import org.criticalsection.mtn4java.commands.invokers.Invoker;
import org.criticalsection.mtn4java.commands.invokers.StandaloneInvoker;
import org.criticalsection.mtn4java.commands.invokers.InvokationException;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

import org.criticalsection.maven.scm.provider.monotone.repository.MonotoneScmProviderRepository;

public class MonotoneStatusCommand extends AbstractStatusCommand
{
    //-----------------------------------------------------------------------------------------
    //  AbstractMonotoneStatusCommand implementation
    //-----------------------------------------------------------------------------------------

    public StatusScmResult executeStatusCommand( ScmProviderRepository repository, ScmFileSet fileSet )
        throws ScmException
    {
        getLogger().debug( "Executing status command" );

        Command cmd = createCommand( repository );

        try
        {
            Invoker invoker = new StandaloneInvoker( "mtn",fileSet.getBasedir().getAbsolutePath() );
            invoker.invoke( cmd );
        }
        catch ( InvokationException e )
        {
            if ( e.getType() == InvokationException.Type.PRECONDITION_ERROR )
            {
                return new StatusScmResult( "No commandline cvailable", "Precondition error during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.PROCESS_CREATION_ERROR )
            {
                return new StatusScmResult( e.getCommandline(), "Error while crreating the external process during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.EXECUTION_ERROR )
            {
                return new StatusScmResult( e.getCommandline(), "Error during command execution.", e.getFormatedErrorMessage(), false );
            }
        }

        List repositoryStatus = new ArrayList();
        Map result = (Map) cmd.getResult();

        for ( Iterator i = result.keySet().iterator(); i.hasNext(); )
        {
            String curKey = (String) i.next();
            InventoryInformation curInfo = (InventoryInformation) result.get( curKey );

            // nur Dateien
            if ( isFile( curInfo ) )
            {
                ScmFile file = addFile( curInfo );
                if ( file != null ) repositoryStatus.add( file );
            }
        }
        
        return new StatusScmResult( repositoryStatus, new ScmResult( null, null, null, true ) );
    }

    //-----------------------------------------------------------------------------------------
    //  MonotoneStatusCommand implementation
    //-----------------------------------------------------------------------------------------

    Command createCommand( ScmProviderRepository repository )
        throws ScmRepositoryException
    {
        MonotoneScmProviderRepository repo = ((MonotoneScmProviderRepository)repository);

        // Create base command
        InventoryCommand cmd = new InventoryCommand();

        // Set base options
        if ( repo.getConfigDirectory() != null && StringUtils.isNotEmpty( repo.getConfigDirectory() ) )
        {
            cmd.setConfigDirPath( repo.getConfigDirectory() );
        }

        if ( repo.getKeyringDirectory() != null && StringUtils.isNotEmpty( repo.getKeyringDirectory() ) )
        {
            cmd.setKeyDirPath( repo.getKeyringDirectory() );
        }

        if ( repo.getKey() != null && StringUtils.isNotEmpty( repo.getKey() ) )
        {
            cmd.setKey( repo.getKey() );
        }

        return cmd;
    }

    private boolean isFile( InventoryInformation info )
    {
        return info.getNewType() == InventoryInformation.Type.FILE || info.getFilesystemType() == InventoryInformation.FsType.FILE;
    }


    private ScmFile addFile( InventoryInformation info )
    {
        ScmFile result = null;

        // Added
        if ( info.getStatus().contains( InventoryInformation.Status.ADDED ) )
        {
            result = new ScmFile( info.getPath(), ScmFileStatus.ADDED );
        }
        // Deleted
        else if ( info.getStatus().contains( InventoryInformation.Status.DROPPED ) )
        {
            result= new ScmFile( info.getPath(), ScmFileStatus.DELETED );
        }
        //modified
        else if ( info.getStatus().contains( InventoryInformation.Status.RENAME_SOURCE ) ||
             info.getStatus().contains( InventoryInformation.Status.RENAME_TARGET ) ||
             info.getChanges().contains( InventoryInformation.ChangeType.CONTENT ) ||
             info.getChanges().contains( InventoryInformation.ChangeType.ATTRIBUTES )
            )
        {
            result = new ScmFile( info.getPath(), ScmFileStatus.MODIFIED );
        }

        return result;
    }

}
